/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "Resource.h"

#include <assert.h>
#include <iostream>

namespace Engine {
namespace Content {


Resource::Resource()
{
    m_thisReadyToUse = false;
    m_numReferences = 0;
    m_numChildResourcesNotReady = 0;
}

Resource::~Resource()
{
    assert(m_parentResources.size() == 0);
    for (unsigned i = 0; i < m_childResources.size(); i++) {
        swapRemoveSingle<Resource*>(m_childResources[i]->m_parentResources, this);
    }
}

void Resource::setReady()
{
    checkCounter();
    if (m_thisReadyToUse)
        return;

    m_thisReadyToUse = true;
    if (childrenReadyToUse())
        becomeReady();
    checkCounter();
}

void Resource::setNotReady()
{
    checkCounter();
    if (!m_thisReadyToUse)
        return;

    m_thisReadyToUse = false;
    if (childrenReadyToUse())
        becomeNotReady();
    checkCounter();
}


void Resource::addChildResource(Resource *resource)
{
    checkCounter();
    m_childResources.push_back(resource);
    resource->m_parentResources.push_back(this);
    if (!resource->readyToUse()) {
        m_numChildResourcesNotReady++;
    }
    checkCounter();
}

void Resource::removeChildResource(Resource *resource)
{
    checkCounter();
    swapRemoveSingle<Resource*>(resource->m_parentResources, this);
    swapRemoveSingle<RefCountPtr<Resource>>(m_childResources, RefCountPtr<Resource>(resource));

    if (!resource->readyToUse()) {
        m_numChildResourcesNotReady--;
        if (readyToUse()) {
            becomeReady();
        }
    }
    checkCounter();
}

void Resource::removeAllChildResources()
{
    checkCounter();
    for (unsigned i = 0; i < m_childResources.size(); i++) {
        swapRemoveSingle<Resource*>(m_childResources[i]->m_parentResources, this);
    }
    m_childResources.clear();

    m_numChildResourcesNotReady = 0;
    if (readyToUse()) {
        becomeReady();
    }
    checkCounter();
}


void Resource::onAllChildResourcesReady()
{

}

void Resource::onNotAllChildResourceReady()
{

}


void Resource::becomeReady()
{
    for (unsigned i = 0; i < m_parentResources.size(); i++) {
        m_parentResources[i]->childResourceBecameReady();
    }
}

void Resource::becomeNotReady()
{
    for (unsigned i = 0; i < m_parentResources.size(); i++) {
        m_parentResources[i]->childResourceBecameNotReady();
    }
}

void Resource::childResourceBecameReady()
{
    m_numChildResourcesNotReady--;
    checkCounter();
    if (childrenReadyToUse()) {
        bool thisWasReady = m_thisReadyToUse;
        onAllChildResourcesReady();
        if (thisWasReady)
            becomeReady();
    }
    checkCounter();
}

void Resource::childResourceBecameNotReady()
{
    bool allChildrenReady = childrenReadyToUse();
    m_numChildResourcesNotReady++;
    checkCounter();
    if (allChildrenReady) {
        onNotAllChildResourceReady();
        if (m_thisReadyToUse)
            becomeNotReady();
    }
    checkCounter();
}

void Resource::checkCounter()
{
    unsigned numNotReady = 0;
    for (unsigned i = 0; i < m_childResources.size(); i++) {
        if (!m_childResources[i]->readyToUse())
            numNotReady++;
    }
    if (numNotReady != m_numChildResourcesNotReady) {
        std::cout << "m_numChildResourcesNotReady corrupted!" << std::endl;
        std::cout << typeid(this).name() << std::endl;
        assert(false && "m_numChildResourcesNotReady corrupted!");
    }
}


}
}
