/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "GLSLProgram.h"
#include "GLSLShader.h"

#include <GL/glew.h>

namespace Engine {
namespace Graphics {

GLSLProgram::GLSLProgram(RenderDevice *renderDevice)
{
    m_programID = glCreateProgram();
}

GLSLProgram::~GLSLProgram()
{
    glDeleteProgram(m_programID);
}


void GLSLProgram::attach(GLSLShader *shader)
{
    glAttachShader(m_programID, shader->getID());
}

bool GLSLProgram::link()
{
    glLinkProgram(m_programID);
    int success;
    glGetProgramiv(m_programID, GL_LINK_STATUS, &success);
    return success == GL_TRUE;
}

std::string GLSLProgram::getLog()
{
    int logLength;
    glGetProgramiv(m_programID, GL_INFO_LOG_LENGTH, &logLength);
    if (logLength <= 0)
        return "";
    char buff[logLength];
    glGetProgramInfoLog(m_programID, logLength, NULL, buff);
    return buff;
}

void GLSLProgram::clearProgram()
{
    glDeleteProgram(m_programID);
    m_programID = glCreateProgram();
    setNotReady();
}


void GLSLProgram::bindUniformBlockToSlot(unsigned index, unsigned slot)
{
    glUniformBlockBinding(m_programID, index, slot);
}

unsigned GLSLProgram::getUniformLocation(const char *name)
{
    return glGetUniformLocation(m_programID, name);
}

unsigned GLSLProgram::getUniformBlockIndex(const char *name)
{
    return glGetUniformBlockIndex(m_programID, name);
}

void GLSLProgram::getUniformBlockLayout(UniformBlockLayout &layout)
{

    int uniformBlockSize;
    glGetActiveUniformBlockiv(m_programID, getUniformBlockIndex(layout.m_name),
                              GL_UNIFORM_BLOCK_DATA_SIZE, &uniformBlockSize);

    const char* names[layout.m_entries.size()];
    for (unsigned i = 0; i < layout.m_entries.size(); i++)
        names[i] = layout.m_entries[i].m_name;

    unsigned indices[layout.m_entries.size()];
    glGetUniformIndices(m_programID, layout.m_entries.size(), names, indices);

    int offset[layout.m_entries.size()];
    glGetActiveUniformsiv(m_programID, layout.m_entries.size(), indices,
                          GL_UNIFORM_OFFSET, offset);



    int size[layout.m_entries.size()];
    glGetActiveUniformsiv(m_programID, layout.m_entries.size(), indices,
                              GL_UNIFORM_SIZE, size);

    int matrixStride[layout.m_entries.size()];
    glGetActiveUniformsiv(m_programID, layout.m_entries.size(), indices,
                          GL_UNIFORM_MATRIX_STRIDE, matrixStride);



    layout.m_blockSize = uniformBlockSize;
    for (unsigned i = 0; i < layout.m_entries.size(); i++) {
        layout.m_entries[i].m_valid = offset[i] >= 0;
        layout.m_entries[i].m_offset = offset[i];
        layout.m_entries[i].m_numElements = size[i];
        layout.m_entries[i].m_matrixStride = matrixStride[i];
    }

}


}
}
