/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "Texture1D.h"
#include "RenderDevice.h"
#include <algorithm>


namespace Engine {
namespace Graphics {

Texture1D::Texture1D(RenderDevice *renderDevice) : BaseTexture(renderDevice)
{
    m_width = 0;
    m_levels = 0;
    m_format = (TextureFormat)0;
}



void Texture1D::resize(unsigned width, unsigned levels, TextureFormat format)
{
    if ((width == m_width) && (levels == m_levels) && (format == m_format))
        return;

    m_renderDevice->bindTexture1D(this);

    m_width = width;
    m_format = format;
    m_levels = levels;
#if 1
    int f, t;
#define GL_TEXTURE_IMAGE_FORMAT           0x828F
#define GL_TEXTURE_IMAGE_TYPE             0x8290
    glGetInternalformativ(GL_TEXTURE_1D, m_format, GL_TEXTURE_IMAGE_FORMAT, 1, &f);
    glGetInternalformativ(GL_TEXTURE_1D, m_format, GL_TEXTURE_IMAGE_TYPE, 1, &t);
    if (f == 0)
        f = GL_RGBA;


    for (unsigned i = 0; i < m_levels; i++)
        glTexImage1D(GL_TEXTURE_1D, i, m_format, std::max(1u, width >> i), 0, f, t, NULL);
#else
    glTexStorage2D(GL_TEXTURE_2D, m_levels, m_format, m_width, m_height);

#endif
    glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MAX_LEVEL, m_levels-1);

    glTexParameterf(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameterf(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameterf(GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
}

void Texture1D::uploadTextureLevel(unsigned level, const void *data, TextureFormat sourceFormat)
{
    m_renderDevice->bindTexture1D(this);

    unsigned w = std::max(1u, m_width >> level);
    int f, t;
#define GL_TEXTURE_IMAGE_FORMAT           0x828F
#define GL_TEXTURE_IMAGE_TYPE             0x8290
    glGetInternalformativ(GL_TEXTURE_1D, sourceFormat, GL_TEXTURE_IMAGE_FORMAT, 1, &f);
    glGetInternalformativ(GL_TEXTURE_1D, sourceFormat, GL_TEXTURE_IMAGE_TYPE, 1, &t);
    glTexSubImage1D(GL_TEXTURE_1D, level, 0, w, f, t, data);
}

void Texture1D::setupFilter(unsigned mag, unsigned min)
{
    m_renderDevice->bindTexture1D(this);
    glTexParameterf(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, min);
    glTexParameterf(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, mag);
}

void Texture1D::setupWrapMode(unsigned wrapS)
{
    m_renderDevice->bindTexture1D(this);
    glTexParameterf(GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, wrapS);
}

}
}
