/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "X11Window.h"
#include <stdexcept>
#include <iostream>
#include <stdio.h>
#include <string.h>

#include <X11/Xatom.h>

namespace Engine {
namespace Platform {


X11RenderContext::X11RenderContext(X11Window *window) : m_window(window)
{
    m_context = 0;
}

X11RenderContext::~X11RenderContext()
{

}


void X11RenderContext::makeCurrent()
{
    m_window->makeCurrent();
}

void X11RenderContext::swap()
{
    m_window->swap();
}


X11Window::X11Window()
{
    m_renderContext = new X11RenderContext(this);
    m_display = NULL;
    m_glxExtensionString = NULL;
    m_window = 0;
    m_colormap = 0;
    openDisplay();
}

X11Window::~X11Window()
{
    freeWindow();
    closeDisplay();
}

void X11Window::resetWindow(WindowParams &params)
{
    freeWindow();
    m_params = params;
    createWindow();
}


void X11Window::openDisplay()
{
    m_display = XOpenDisplay(0);
    if (m_display == NULL)
        throw std::runtime_error("Could not open X11 display!");

    int glxMajor, glxMinor;

    // FBConfigs were added in GLX version 1.3.
    if ((!glXQueryVersion(m_display, &glxMajor, &glxMinor)) ||
        ((glxMajor == 1) && (glxMinor < 3)) || (glxMajor < 1))
        throw std::runtime_error("At least glx version 1.3 is required!");


    m_glxExtensionString = glXQueryExtensionsString(m_display, DefaultScreen(m_display));
  //  std::cout << m_glxExtensionString << std::endl;
}

void X11Window::closeDisplay()
{
    if (m_display != NULL)
        XCloseDisplay(m_display);
    m_display = NULL;
}


#define GLX_CONTEXT_MAJOR_VERSION_ARB       0x2091
#define GLX_CONTEXT_MINOR_VERSION_ARB       0x2092
typedef GLXContext (*glXCreateContextAttribsARBProc)(Display*, GLXFBConfig, GLXContext, Bool, const int*);

void X11Window::createWindow()
{

    int fbCount;
    GLXFBConfig *fbc = glXChooseFBConfig(m_display, DefaultScreen(m_display),
                                        m_params.visualAttributes.empty()?NULL:&m_params.visualAttributes[0], &fbCount);
    if ((fbc == NULL) || (fbCount == 0))
        throw std::runtime_error("Could not find a compatible X11 frame buffer config!");

    GLXFBConfig bestFbc = fbc[0];
    XVisualInfo *vi = glXGetVisualFromFBConfig(m_display, bestFbc);
    XFree(fbc);

    if (vi == NULL)
        throw std::runtime_error("Could not find a compatible X11 visual!");


    XSetWindowAttributes swa;
    swa.colormap = m_colormap = XCreateColormap(m_display,
                                     RootWindow(m_display, vi->screen),
                                     vi->visual, AllocNone);
    swa.background_pixmap = None;
    swa.border_pixel      = 0;
    swa.event_mask        = StructureNotifyMask | KeyPressMask | KeyReleaseMask | ButtonPressMask | ButtonReleaseMask | PointerMotionMask;

    m_window = XCreateWindow(m_display, RootWindow(m_display, vi->screen),
                              0, 0, m_params.width, m_params.height, 0, vi->depth, InputOutput,
                              vi->visual,
                              CWBorderPixel|CWColormap|CWEventMask, &swa);

    if (m_window == 0) {
        XFree(vi);
        XFreeColormap(m_display, m_colormap);
        m_colormap = 0;
        throw std::runtime_error("Could not open window!");
    }

    XFree(vi);

    XStoreName(m_display, m_window, m_params.windowName.c_str());

    XMapWindow(m_display, m_window);

    if (m_params.fullscreen) {
        Atom wm_state = XInternAtom(m_display, "_NET_WM_STATE", False);
        Atom fullscreen = XInternAtom(m_display, "_NET_WM_STATE_FULLSCREEN", False);

        #if 1
        XEvent xev;
        memset(&xev, 0, sizeof(xev));
        xev.type = ClientMessage;
        xev.xclient.window = m_window;
        xev.xclient.message_type = wm_state;
        xev.xclient.format = 32;
        xev.xclient.data.l[0] = 1;
        xev.xclient.data.l[1] = fullscreen;
        xev.xclient.data.l[2] = 0;

        XSendEvent(m_display, DefaultRootWindow(m_display), False,
                    SubstructureRedirectMask | SubstructureNotifyMask, &xev);
        #else
        Atom atoms[1];
        atoms[0] = fullscreen;
        XChangeProperty(m_display, m_window, wm_state, XA_ATOM, 32, PropModeReplace, (unsigned char*)atoms, 1);
        #endif

        if (m_params.grabInput) {
            XWarpPointer(m_display, None, m_window, 0, 0, 0, 0, 0, 0);
            XGrabKeyboard(m_display, m_window, True, GrabModeAsync,
                GrabModeAsync, CurrentTime);
            XGrabPointer(m_display, m_window, True, ButtonPressMask,
                GrabModeAsync, GrabModeAsync, m_window, None, CurrentTime);
        }

    }

    Atom wmdelete = XInternAtom(m_display, "WM_DELETE_WINDOW", True);
    XSetWMProtocols(m_display, m_window, &wmdelete, 1);



    glXCreateContextAttribsARBProc glXCreateContextAttribsARB = (glXCreateContextAttribsARBProc) glXGetProcAddressARB( (const GLubyte *) "glXCreateContextAttribsARB" );

    if (glXCreateContextAttribsARB == NULL)
        throw std::runtime_error("glXCreateContextAttribsARB is not supported!");


    context()->m_context = glXCreateContextAttribsARB(m_display, bestFbc, 0, True, m_params.contextAttributes.empty()?NULL:&m_params.contextAttributes[0]);

    if (context()->m_context == 0) {
        throw std::runtime_error("Could not create context!");
    }

    makeCurrent();

}

bool X11Window::getNextEvent(Event &event)
{
    while (XPending(m_display) > 0) {

        XEvent xev;
        memset(&xev, 0, sizeof(XEvent));
        XNextEvent(m_display, &xev);

        switch (xev.type) {
            case ConfigureNotify:
                if (((int)m_params.width != xev.xconfigure.width) || ((int)m_params.height != xev.xconfigure.height)) {
                    m_params.width = xev.xconfigure.width;
                    m_params.height = xev.xconfigure.height;
                    event.notification.type = Event::TYPE_NOTIFICATION;
                    event.notification.message = Event::Notification::NM_RESIZE;
                    return true;
                }
            break;

            case KeyPress: {
                unsigned keysym = XLookupKeysym(&xev.xkey, 0);
                event.key.type = Event::TYPE_KEY_DOWN;
                event.key.keysym = keysym;
                event.key.x = xev.xkey.x;
                event.key.y = xev.xkey.y;
                return true;
            } break;

            case KeyRelease: {
                unsigned keysym = XLookupKeysym(&xev.xkey, 0);
                event.key.type = Event::TYPE_KEY_UP;
                event.key.keysym = keysym;
                event.key.x = xev.xkey.x;
                event.key.y = xev.xkey.y;
                return true;
            } break;


            case MappingNotify:
                XRefreshKeyboardMapping(&xev.xmapping);
            break;

            case MotionNotify:
                event.motion.type = Event::TYPE_MOTION;
                event.motion.x = xev.xmotion.x;
                event.motion.y = xev.xmotion.y;

                event.motion.relX = xev.xmotion.x - m_state.mouseX;
                event.motion.relY = xev.xmotion.y - m_state.mouseY;

                m_state.mouseX = xev.xmotion.x;
                m_state.mouseY = xev.xmotion.y;
                m_state.mouseRelX = event.motion.relX;
                m_state.mouseRelX = event.motion.relY;

                /*

                if (!xev.xmotion.send_event) {
                    std::cout << "Mouse at " << xev.xmotion.x << "  " << xev.xmotion.y << std::endl;
                    XWarpPointer(m_display, m_window, m_window, 0, 0, 0, 0, m_params.width/2, m_params.height/2);
                }
                */
                return true;
            break;

            case ButtonPress:
                m_state.buttonState |= 1 << xev.xbutton.button;

                event.button.type = Event::TYPE_BUTTON_DOWN;
                event.button.index = xev.xbutton.button;
                event.button.x = xev.xbutton.x;
                event.button.y = xev.xbutton.y;
                return true;
            break;

            case ButtonRelease:
                m_state.buttonState &= ~(1 << xev.xbutton.button);

                event.button.type = Event::TYPE_BUTTON_UP;
                event.button.index = xev.xbutton.button;
                event.button.x = xev.xbutton.x;
                event.button.y = xev.xbutton.y;
                return true;
            break;

            case ClientMessage:
                if (strcmp(XGetAtomName(m_display, xev.xclient.message_type),
                           "WM_PROTOCOLS") == 0) {

                    if ((xev.xclient.format == 32) && (xev.xclient.data.l[0] == (unsigned)XInternAtom(m_display, "WM_DELETE_WINDOW", false))) {
                        m_state.shutdown = true;
                        event.notification.type = Event::TYPE_NOTIFICATION;
                        event.notification.message = Event::Notification::NM_WMSHUTDOWN;
                        return true;
                    }
                }
            break;
        }
    }
    return false;
}



void X11Window::freeWindow()
{
    if (context()->m_context != 0) {
        glXMakeCurrent(m_display, 0, 0);
        glXDestroyContext(m_display, context()->m_context);
        context()->m_context = 0;
    }

    if (m_window != 0) {
        XDestroyWindow(m_display, m_window);
        m_window = 0;
    }
    if (m_colormap != 0) {
        XFreeColormap(m_display, m_colormap);
        m_colormap = 0;
    }
}

void X11Window::makeCurrent()
{
    glXMakeCurrent(m_display, m_window, context()->m_context);
}

void X11Window::swap()
{
    glXSwapBuffers(m_display, m_window);
}

}
}
