/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef SINGLETRACKBUNDLEADJUSTMENT_H
#define SINGLETRACKBUNDLEADJUSTMENT_H

#include <vector>

#include <immintrin.h>
#include "../../tools/LinAlg.h"
#include "../../tools/AlignedAllocator.h"

#include "PFBundleAdjustment.h"

namespace SFM {

/**
 * @brief Performs bundle adjustment but restricts the updates to the tracks' positions.
 *
 * @ingroup BundleAdjustment_Group
 */
class SingleTrackBundleAdjustment
{
    public:
        struct Observation {
            const LinAlg::Matrix3x4f *projectionViewMatrix;
            LinAlg::Vector2f screenSpacePosition;
            float weight;

            const PFBundleAdjustment::PFBundleAdjustment::RadialDistortionParametrization *distortion;
        };

        /// Optimizes the position of a single track for the given observations of the track.
        void optimize(const std::vector<Observation> &observations, LinAlg::Vector4f &trackPosition);
    protected:

        void computeResiduals(const std::vector<Observation> &observations, const LinAlg::Vector4f &trackPosition);
        void computeJacobian(const std::vector<Observation> &observations, const LinAlg::Vector4f &trackPosition);

        float sumResiduals();

        std::vector<float> m_residuals;
        std::vector<LinAlg::Vector4f, AlignedAllocator<LinAlg::Vector4f> > m_jacobian;

        struct Hessian {
            __m128 H0, H1, H2, H3;

            void computeHessianFromJacobi(const std::vector<LinAlg::Vector4f, AlignedAllocator<LinAlg::Vector4f> > &jacobian, float lambda);

            __m128 operator*(const __m128 &v) const;
            __m128 getDiagonal() const;
        };
};

}

#endif // SINGLETRACKBUNDLEADJUSTMENT_H
