/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef DUPLICATETRACKREMOVAL_H
#define DUPLICATETRACKREMOVAL_H

#include "../tools/ChunkedArray.hpp"
#include "../tools/LinAlg.h"
#include "../tools/AABB.h"

#include <vector>
#include <memory>

#include <assert.h>

namespace SFM {

/**
 * @brief Hash based data structure to accelerate the search for duplicate tracks.
 * @details Whenever duplicates are searched for, this data structure is recreated.
 * tracks are added into it one by one. For every track, the data structure first
 * checks for duplicates and either returns a pointer to the duplicate or adds
 * the new track.
 * @ingroup SFMBackend_Group
 */
class DuplicateTrackRemoval
{
    public:
        DuplicateTrackRemoval();

        struct Element {
            unsigned m_userData;
            LinAlg::Vector3f m_position;
            float m_size;
        };

        /// Clears the data structure
        void clear();

        /**
         * @brief Searches for a duplicate of the passed in element .
         * @details If one is found, a pointer to the duplicate is returned. Otherwise the new element
         * is added into the hash map and a null pointer is returned.
         */
        Element *mergeOrAdd(const Element &element);
    protected:

        struct HashCell {
            ChunkedArray<Element> m_elements;
        };

        template<uint64_t size, int64_t facX, int64_t facY, int64_t facZ, int64_t facScale>
        struct Hash {
            HashCell cells[size];

            void reset() {
                for (unsigned i = 0; i < size; i++)
                    cells[i].m_elements.clear();
            }

            static unsigned computeHash(int64_t x, int64_t y, int64_t z, int64_t scale) {
                return uint64_t(x * facX + y * facY + z * facZ + scale * facScale) % size;
            }

            HashCell &operator()(int64_t x, int64_t y, int64_t z, int64_t scale) { return cells[computeHash(x, y, z, scale)]; }
            const HashCell &operator()(int64_t x, int64_t y, int64_t z, int64_t scale) const { return cells[computeHash(x, y, z, scale)]; }
        };

        typedef Hash<(1<<20)-1, 1, 8374560387463l, 23969419341l, 234713456239456l> SizedHash;


        std::unique_ptr<SizedHash> m_hash;
};

}


#endif // DUPLICATETRACKREMOVAL_H
