/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "Frame.h"

#include <string.h>

#include <assert.h>

#include "InternalCameraCalibration.h"
#include "BundleAdjustment/BundleAdjustment.h"

#include <set>
#include <map>
#include <iomanip>

#include "../tools/PCVToolbox.hpp"

#include <boost/random.hpp>
#include <chrono>

#include "Track.h"
#include "ViewMatrixEstimator.h"
#include "../cudaUtilities/cudaProfilingScope.h"
#include "SFM.h"

namespace SFM {


Frame::Frame(InternalCameraCalibration *internalCameraCalibration, const std::string &filename, unsigned frameIndex) :
    m_frameIndex(frameIndex), m_imageFilename(filename), m_camera(internalCameraCalibration), m_siftDescriptorDB(5, 10),
    m_observationGrid(10)
{
    m_width = 0;
    m_height = 0;
    m_active = false;
}

Frame::~Frame()
{
    //dtor
}

void Frame::setInitialFeaturePoints(const CudaSift::FeaturePoint *featurePoints, unsigned count, unsigned width, unsigned height, unsigned numFrames, bool testbenchDontCompile)
{
    if (height/(float)width != m_camera.getInternalCalibration()->getAspectRatio()) {
        std::stringstream msg;
        msg << "Apect ratio of image " << m_imageFilename << " is different from the image aspect ratio specified for the internal calibration. This is probably a mistake!";
        throw std::runtime_error(msg.str());
    }
    AddCudaScopedProfileInterval("Frame::setInitialFeaturePoints");

    m_width = width;
    m_height = height;

    m_featurePoints.resize(count);

    std::vector<SiftDescriptorDB::SiftDescriptor> dbDescriptors;
    std::vector<LinAlg::Vector2f> screenSpacePositions;
    dbDescriptors.resize(count);
    screenSpacePositions.resize(count);

    for (unsigned i = 0; i < m_featurePoints.size(); i++) {
        const CudaSift::FeaturePoint &srcFP = featurePoints[i];
        m_featurePoints[i].x = srcFP.x / width * 2.0f - 1.0f;
        m_featurePoints[i].y = (srcFP.y / height * 2.0f - 1.0f) * m_camera.getInternalCalibration()->getAspectRatio();

    float scale = srcFP.scale * 0.5f;
        int ilod = std::log2(scale);
        m_featurePoints[i].lod = ilod+  scale / (1 << ilod) - 1.0f;

        m_featurePoints[i].size = scale / (float)width;
        m_featurePoints[i].angle = srcFP.angle;
        m_featurePoints[i].locationPrecision = srcFP.locationPrecision;

        dbDescriptors[i].userData = i;
        dbDescriptors[i].projectionIndex = -1;
        memcpy(dbDescriptors[i].descriptor, featurePoints[i].descriptor, 128);

        m_observationGrid.getCell(LinAlg::Fill(m_featurePoints[i].x, m_featurePoints[i].y)).patches.push_back(i);


        screenSpacePositions[i][0] = featurePoints[i].x / width;
        screenSpacePositions[i][1] = featurePoints[i].y / height;


#ifdef CudaSift_EXTRACT_PATCH_DATA
        m_featurePoints[i].packedPatch = featurePoints[i].packedPatch;
#endif
    }

    if (!testbenchDontCompile) {
        {
            AddCudaScopedProfileInterval("Compiling descriptors");
            m_siftDescriptorDB.compile(&dbDescriptors[0], dbDescriptors.size());
        }
        {
            AddCudaScopedProfileInterval("Computing uniquenesses");
            m_siftDescriptorDB.recomputeUniquenesses(screenSpacePositions);
        }

        for (unsigned i = 0; i < m_siftDescriptorDB.getNumDescriptors(); i++) {
            m_featurePoints[m_siftDescriptorDB.getDescriptor(i).userData].siftDBIndex = i;
        }

#define SORT_FEATUREPOINTS_BY_DB
#ifdef SORT_FEATUREPOINTS_BY_DB
        std::vector<ImageFeaturePoint> tmp;
        tmp.resize(m_featurePoints.size());
        for (unsigned i = 0; i < m_siftDescriptorDB.getNumDescriptors(); i++) {
            tmp[i] = m_featurePoints[m_siftDescriptorDB.getDescriptor(i).userData];
            m_siftDescriptorDB.setUserData(i, i);
        }
        m_featurePoints.swap(tmp);
#endif

    }
    m_interFrameRelations.resize(numFrames, InterFrameRelation(m_featurePoints.size()));
}


void Frame::ransacEstimateViewMatrixFromTracks(ChunkedArray<Track> &trackList, LinAlg::Matrix4x4f &viewMatrix, float &score,
                                               const std::vector<SiftDescriptorDB::SiftDescriptor> &globalPatchDescrList,
                                               config::FeatureMatchingConfig &matchingConfig,
                                               DescriptorMatchingStats &descriptorMatchingStats)
{
    /*
    for (unsigned i = 0; i < m_featurePoints.size(); i++) {
        ImageFeaturePoint &fp = m_featurePoints[i];
        fp.bestTrackCandidate = -1;
    }
    */

    std::vector<ViewMatrixEstimator::ControlPoint> controlPoints;

    std::vector<float> similarityScores;

    std::map<uint64_t, unsigned> trackToCandidateMap;

    std::vector<SiftDescriptorDB::Match> matches;

    descriptorMatchingStats.startMatching();
    {
        AddCudaScopedProfileInterval("Adding frame phase 1 DB lookup");
        m_siftDescriptorDB.findMatches(&globalPatchDescrList[0], globalPatchDescrList.size(), matches, matchingConfig.matchingMultithreadingBatchSize);
    }
    descriptorMatchingStats.finishedMatching(DescriptorMatchingStats::MATCH_UNCONSTRAINED, globalPatchDescrList.size());


    for (unsigned j = 0; j < matches.size(); j++) {
        if ((matches[j].normalizedDiff < matchingConfig.frameNewCameraMatchingConfig.maxNormalizedMatchDiff) &&
            (matches[j].closestMatchIndex != (unsigned)-1)) {

            const unsigned trackIndex = globalPatchDescrList[matches[j].srcIndex].userData;
            Track &track = trackList[trackIndex];

            unsigned featurePointIndex = m_siftDescriptorDB.getDescriptor(matches[j].closestMatchIndex).userData;
            auto it = trackToCandidateMap.find(((uint64_t)trackIndex << 32ul) | featurePointIndex);
            if (it == trackToCandidateMap.end()) {
                trackToCandidateMap[((uint64_t)trackIndex << 32ul) | featurePointIndex] = controlPoints.size();

                ImageFeaturePoint &featurePoint = m_featurePoints[featurePointIndex];

                ViewMatrixEstimator::ControlPoint controlPoint;
                controlPoint.index[0] = trackIndex;
                controlPoint.index[1] = featurePointIndex;
                controlPoint.worldSpacePosition = track.getLastWSPositionEstimate();
                controlPoint.screenSpacePosition = LinAlg::Fill(featurePoint.x, featurePoint.y);
                controlPoint.rcpSqrScreenSize = 1.0f / (featurePoint.size*featurePoint.size);
                controlPoint.normalizedDifference = matches[j].normalizedDiff;

                controlPoints.push_back(controlPoint);

                //featurePoint.bestTrackCandidate = trackIndex;
            } else {
                controlPoints[it->second].normalizedDifference = fmin(controlPoints[it->second].normalizedDifference, matches[j].normalizedDiff);
            }
        }
    }

    if (matchingConfig.verbatimLevel >= 2) {
        std::cout << "ransacEstimateViewMatrixFromTracks: initial matches: " << controlPoints.size() << std::endl;
    }

    if (controlPoints.size() < matchingConfig.frameNewCameraMatchingConfig.minInitialMatchCount) {
        score = 0.0f;
        return;
    }

    ViewMatrixEstimator viewMatrixEstimator;

    float bestScore = 0.0f;
    LinAlg::Matrix4x4f bestViewMatrix;
    LinAlg::Matrix3x4f calibrationMatrix = m_camera.getInternalCalibration()->getProjectionMatrix().dropRow(2);
    {
        AddCudaScopedProfileInterval("Adding frame phase 1 RANSAC");
        viewMatrixEstimator.estimate(calibrationMatrix, &controlPoints[0], NULL, controlPoints.size(),
                                matchingConfig.frameNewCameraMatchingConfig.numInitialRansacIterations,
                                bestViewMatrix, bestScore);
    }

    viewMatrix = bestViewMatrix;
    score = bestScore;
}


void Frame::activate(ChunkedArray<Track> &trackList, const LinAlg::Matrix4x4f &viewMatrix, BundleAdjustment &ba)
{
    m_camera.activateBA(ba);
    m_camera.setViewMatrix(viewMatrix);
    m_camera.updateBA(ba);

    setActive();
}


struct NewObsMatcher : public SiftDescriptorDB::MatchConstraint
{
    std::vector<const SiftDescriptorDB::SiftDescriptor*> descriptors;
    std::vector<unsigned> trackIndex;
    std::vector<float> cachedDistances;

    struct Track {
        unsigned trackIndex;
        LinAlg::Vector2f screenPos;
        float bestScore;
        float normalizedDiff;
        unsigned bestIndex;
        unsigned lock;
    };
    std::vector<Track> tracks;

    Frame *frame;


    float maxMatchError;
    float maxSQRDistanceError;

    void prepare() {
        for (unsigned i = 0; i < tracks.size(); i++) {
            tracks[i].bestScore = -1.0f;
            tracks[i].bestIndex = -1;
            tracks[i].lock = 0;
        }
        cachedDistances.resize(descriptors.size());
    }


    virtual unsigned getNumDescriptors() const { return descriptors.size(); }
    virtual const SiftDescriptorDB::SiftDescriptor **getDescriptors() { return &descriptors[0]; }

    virtual bool preMatchTest(unsigned sourceIndex, unsigned destinationIndex) {
#ifdef SORT_FEATUREPOINTS_BY_DB
        const Frame::ImageFeaturePoint &targetFeaturePoint = frame->getFeaturePoint(destinationIndex);
#else
        const SiftDescriptorDB::SiftDescriptor &targetDescriptor = frame->getSiftDescriptorDB().getDescriptor(destinationIndex);
        const Frame::ImageFeaturePoint &targetFeaturePoint = frame->getFeaturePoint(targetDescriptor.userData);
#endif

        const Track &track = tracks[trackIndex[sourceIndex]];

        float distance = (LinAlg::Fill(targetFeaturePoint.x, targetFeaturePoint.y) - track.screenPos).SQRLen();
        distance /= targetFeaturePoint.size * targetFeaturePoint.size;
        cachedDistances[sourceIndex] = distance;

        return distance < maxSQRDistanceError;
    }
    virtual void postMatchOperation(unsigned sourceIndex, unsigned destinationIndex, float normalizedMatchScore) {
        if (normalizedMatchScore < maxMatchError) {
            float distance = cachedDistances[sourceIndex];
            float score = std::exp(-(distance * 3.0f / maxSQRDistanceError + normalizedMatchScore * 3.0f / maxMatchError));

            Track &track = tracks[trackIndex[sourceIndex]];
            while (__sync_lock_test_and_set(&track.lock, 1) == 0) ;

            if (track.bestScore < score) {
                track.bestScore = score;
                track.normalizedDiff = normalizedMatchScore;
                track.bestIndex = destinationIndex;
            }
            __sync_lock_release(&track.lock, 0);
        }
    }

};


void Frame::findNewObservations(ChunkedArray<Track> &trackList, std::vector<NewObservationCandidate> &candidates,
                               unsigned timestampClock, DescriptorMatchingStats &descriptorMatchingStats,
                               config::FeatureMatchingConfig &matchingConfig, bool bootstrappingNewImage)
{
    unsigned numNewObsCandidates = 0;

    LinAlg::Matrix3x4f projectionViewMatrix = m_camera.getProjectionViewMatrix().dropRow(2);


    float minScoreThresh;

    NewObsMatcher obsMatcher;
    obsMatcher.frame = this;
    if (bootstrappingNewImage) {
        obsMatcher.maxMatchError = matchingConfig.frameBoostrappingMatchingConfig.maxNormalizedMatchDiff;
        obsMatcher.maxSQRDistanceError = matchingConfig.frameBoostrappingMatchingConfig.maxSqrDistanceError;
        minScoreThresh = matchingConfig.frameBoostrappingMatchingConfig.minScoreThreshold;
    } else {
        obsMatcher.maxMatchError = matchingConfig.frameNewObservationsMatchingConfig.maxNormalizedMatchDiff;
        obsMatcher.maxSQRDistanceError = matchingConfig.frameNewObservationsMatchingConfig.maxSqrDistanceError;
        minScoreThresh = matchingConfig.frameNewObservationsMatchingConfig.minScoreThreshold;
    }

    for (unsigned i = 0; i < trackList.reservedSize(); i++) {
        if (trackList.inUse(i)) {
            Track &track = trackList[i];
            if (track.getTrackMatchingDescriptor() == NULL)
                continue;
            if (track.getState() == Track::STATE_DISABLED)
                continue;

            if (track.getLastMajorChangeTimestamp() < timestampClock) {
                continue;
            }

            bool trackAlreadyObserved = false;
            for (unsigned j = 0; j < track.getObservations().reservedSize(); j++) {
                if (track.getObservations().inUse(j)) {
                    if (track.getObservations()[j].getFrame() == this)
                        trackAlreadyObserved = true;
                }
            }
            if (trackAlreadyObserved)
                continue;

            LinAlg::Vector3f projScreenPos = projectionViewMatrix * track.getLastWSPositionEstimate();
            LinAlg::Vector2f screenPos = projScreenPos.StripHom() / projScreenPos[2];

            if ((screenPos[0] < -1.0f) || (screenPos[0] > 1.0f) ||
                (screenPos[1] < -(float)m_height/m_width) || (screenPos[1] > (float)m_height/m_width))
                continue;

#if 1
            for (unsigned j = 0; j < track.getTrackMatchingDescriptor()->getDescriptors().size(); j++) {
                obsMatcher.descriptors.push_back(&track.getTrackMatchingDescriptor()->getDescriptors()[j]);
                obsMatcher.trackIndex.push_back(obsMatcher.tracks.size());
            }
#else
            LinAlg::Vector3f viewDir;
            if (std::abs(track.getLastWSPositionEstimate()[3]) < 1e-20f)
                viewDir = track.getLastWSPositionEstimate().StripHom();
            else
                viewDir = track.getLastWSPositionEstimate().StripHom() / track.getLastWSPositionEstimate()[3] - m_camera.getCameraPosition();

            viewDir.normalize();

            unsigned bestDescriptor = 0;
            float bestDot = viewDir * track.getTrackMatchingDescriptor()->getViewDirections()[0];

            for (unsigned j = 1; j < track.getTrackMatchingDescriptor()->getDescriptors().size(); j++) {
                float dot = viewDir * track.getTrackMatchingDescriptor()->getViewDirections()[j];
                if (dot > bestDot) {
                    bestDot = dot;
                    bestDescriptor = j;
                }
            }
            obsMatcher.descriptors.push_back(&track.getTrackMatchingDescriptor()->getDescriptors()[bestDescriptor]);
            obsMatcher.trackIndex.push_back(obsMatcher.tracks.size());
#endif

            NewObsMatcher::Track matcherTrack;
            matcherTrack.screenPos = screenPos;
            matcherTrack.trackIndex = i;

            obsMatcher.tracks.push_back(matcherTrack);
        }
    }
    obsMatcher.prepare();
    if (matchingConfig.verbatimLevel >= 2)
        std::cout << "Matching location with " << obsMatcher.descriptors.size() << " tests" << std::endl;
    descriptorMatchingStats.startMatching();
    m_siftDescriptorDB.findMatchesConstrained(&obsMatcher, matchingConfig.matchingMultithreadingBatchSize);
    descriptorMatchingStats.finishedMatching(DescriptorMatchingStats::MATCH_LOCATION, obsMatcher.descriptors.size());

    for (unsigned i = 0; i < obsMatcher.tracks.size(); i++) {
        if (obsMatcher.tracks[i].bestScore > minScoreThresh) {

            NewObservationCandidate candidate;
            candidate.trackIndex = obsMatcher.tracks[i].trackIndex;
            candidate.frame = this;
            candidate.featurePointIndex = m_siftDescriptorDB.getDescriptor(obsMatcher.tracks[i].bestIndex).userData;
            candidate.minNormDifference = obsMatcher.tracks[i].normalizedDiff;

            candidates.push_back(candidate);
            numNewObsCandidates++;
        }
    }

    if (matchingConfig.verbatimLevel >= 2)
        std::cout << "numNewObsCandidates " << numNewObsCandidates << " out of " << obsMatcher.tracks.size() << " matches" << std::endl;
}


struct NewTrackMatcher : public SiftDescriptorDB::MatchConstraint
{
    float maxMatchError;
    float maxDistanceError;

    std::vector<const SiftDescriptorDB::SiftDescriptor*> descriptors;
    std::vector<LinAlg::Vector2f> screenPositions;
    std::vector<LinAlg::Vector3f> epipolarLines;
    Frame *targetFrame;
    const Frame::InterFrameRelation *interFrameRelation;
    std::vector<float> cachedDistances;
    std::vector<std::pair<float, unsigned> > bestMatches;

    void prepare() {
        bestMatches.resize(descriptors.size());
        cachedDistances.resize(descriptors.size());
        for (unsigned i = 0; i < bestMatches.size(); i++) {
            bestMatches[i].first = -1.0f;
        }
    }

    void computeEpipolarLines(const LinAlg::Matrix3x3f &fundamentalMatrix) {
        epipolarLines.resize(screenPositions.size());
        for (unsigned i = 0; i < screenPositions.size(); i++) {
            LinAlg::Vector3f epipolarLine = fundamentalMatrix * screenPositions[i].AddHom(1.0f);
            float len = sqrtf(epipolarLine.StripHom().SQRLen());
            epipolarLines[i] = epipolarLine * (1.0f / len);
        }
    }

    virtual unsigned getNumDescriptors() const { return descriptors.size(); }
    virtual const SiftDescriptorDB::SiftDescriptor **getDescriptors() { return &descriptors[0]; }
    virtual bool preMatchTest(unsigned sourceIndex, unsigned destinationIndex) {
#ifdef SORT_FEATUREPOINTS_BY_DB
        const Frame::ImageFeaturePoint &targetFeaturePoint = targetFrame->getFeaturePoint(destinationIndex);

#else
        const SiftDescriptorDB::SiftDescriptor &targetDescriptor = targetFrame->getSiftDescriptorDB().getDescriptor(destinationIndex);
        const Frame::ImageFeaturePoint &targetFeaturePoint = targetFrame->getFeaturePoint(targetDescriptor.userData);
#endif

        float distance = fabs(LinAlg::Fill(targetFeaturePoint.x, targetFeaturePoint.y, 1.0f) * epipolarLines[sourceIndex]);
        if (distance >= maxDistanceError * targetFeaturePoint.size)
            return false;

        cachedDistances[sourceIndex] = distance / targetFeaturePoint.size;

        if (interFrameRelation->hasFeaturePointPairBeenTriedBefore(sourceIndex, destinationIndex))
            return false;

        return true;
    }
    virtual void postMatchOperation(unsigned sourceIndex, unsigned destinationIndex, float normalizedMatchScore) {
        if (normalizedMatchScore < maxMatchError) {
            float distance = cachedDistances[sourceIndex];
            float score = std::exp(-(distance / maxDistanceError * 3.0f + normalizedMatchScore / maxMatchError * 3.0f));

            if (bestMatches[sourceIndex].first < score) {
                bestMatches[sourceIndex].first = score;
                bestMatches[sourceIndex].second = destinationIndex;
            }
        }
    }
};

void Frame::findNewTracks(ChunkedArray<Track> &trackList, std::vector<std::unique_ptr<Frame> > &frames, BundleAdjustment &ba,
                          std::vector<NewTrackCandidate> &newTrackCandidates,
                          config::FeatureMatchingConfig &matchingConfig,
                          DescriptorMatchingStats &descriptorMatchingStats)
{
    std::vector<unsigned> activeOtherFrames;
    activeOtherFrames.reserve(frames.size());
    for (unsigned i = 0; i < frames.size(); i++) {
        if (frames[i].get() == this)
            continue;

        if (!(frames[i]->active()))
            continue;

        activeOtherFrames.push_back(i);
    }


    std::set<unsigned> patchesActivelyObserved;
    for (unsigned i = 0; i < trackList.reservedSize(); i++) {
        if (trackList.inUse(i)) {
            Track &track = trackList[i];

            if (track.getState() == Track::STATE_DISABLED)
                continue;

            for (unsigned j = 0; j < track.getObservations().reservedSize(); j++) {
                if (track.getObservations().inUse(j)) {
                    if (track.getObservations()[j].getFrame() == this)
                        patchesActivelyObserved.insert(track.getObservations()[j].getFrameFeaturePointIndex());
                }
            }
        }
    }

    assert(m_interFrameRelations.size() == frames.size());
    std::vector<NewTrackMatcher> newTrackMatchers;
    newTrackMatchers.resize(m_interFrameRelations.size());


    for (unsigned i = 0; i < newTrackMatchers.size(); i++) {
        newTrackMatchers[i].maxMatchError = matchingConfig.frameNewTracksMatchingConfig.maxNormalizedMatchDiff;
        newTrackMatchers[i].maxDistanceError = matchingConfig.frameNewTracksMatchingConfig.maxDistanceError;
        newTrackMatchers[i].targetFrame = frames[i].get();
        newTrackMatchers[i].interFrameRelation = &m_interFrameRelations[i];
    }

    std::vector<std::pair<float, unsigned> > patches;
    unsigned gridSubdivs = m_observationGrid.getNumSubdivs();
    for (unsigned j = 0; j < gridSubdivs; j++) {
        for (unsigned i = 0; i < gridSubdivs; i++) {

            ObservationGrid::Cell &cell = m_observationGrid.getCell(i, j);

            unsigned numActiveObs = 0;

            for (LinkedListEntry<TrackObservation> *it = cell.observations.getFirst(); it != NULL; it = it->getNext()) {
                if ((!it->getHost().isFaulty()) && (it->getHost().getTrack()->getState() != Track::STATE_DISABLED))
                    numActiveObs++;
            }

            if ((numActiveObs > matchingConfig.frameNewTracksMatchingConfig.maxNumActiveObsPerCell) ||
                (cell.patches.size() <= 0))
                continue;

            patches.clear();

            for (unsigned k = 0; k < cell.patches.size(); k++) {
                if (patchesActivelyObserved.find(cell.patches[k]) != patchesActivelyObserved.end())
                    continue;

                const ImageFeaturePoint &thisFeaturePoint = m_featurePoints[cell.patches[k]];
                const SiftDescriptorDB::SiftDescriptor &thisFPDesc = m_siftDescriptorDB.getDescriptor(thisFeaturePoint.siftDBIndex);
                if (thisFPDesc.uniqueness < matchingConfig.frameNewTracksMatchingConfig.minObservationUniqueness)
                    continue;

                patches.push_back(std::pair<float, unsigned>(thisFPDesc.uniqueness, cell.patches[k]));
            }
            std::sort(patches.begin(), patches.end());
            for (unsigned k = 0; k < std::min<unsigned>(patches.size(), matchingConfig.frameNewTracksMatchingConfig.maxObservationsTestedPerCellNum); k++) {

                const unsigned featurePointIndex = patches[patches.size()-1-k].second;
                const ImageFeaturePoint &thisFeaturePoint = m_featurePoints[featurePointIndex];
                const SiftDescriptorDB::SiftDescriptor &thisFPDesc = m_siftDescriptorDB.getDescriptor(thisFeaturePoint.siftDBIndex);

                for (unsigned l = 0; l < activeOtherFrames.size(); l++) {
                    unsigned otherFrameIndex = activeOtherFrames[l];
                    if (!m_interFrameRelations[otherFrameIndex].isFeaturePointDry(featurePointIndex)) {
                        newTrackMatchers[otherFrameIndex].descriptors.push_back(&thisFPDesc);
                        newTrackMatchers[otherFrameIndex].screenPositions.push_back(LinAlg::Fill(thisFeaturePoint.x, thisFeaturePoint.y));
                    }
                }
            }
        }
    }

    SiftDescriptorDB::AsyncMatching asyncMatching;

    descriptorMatchingStats.startMatching();
    unsigned numQuerries = 0;
    for (unsigned i = 0; i < newTrackMatchers.size(); i++) {
        if (newTrackMatchers[i].descriptors.empty())
            continue;

        newTrackMatchers[i].prepare();
        newTrackMatchers[i].computeEpipolarLines(PCV::computeFundamental(m_camera.getProjectionViewMatrix().dropRow(2),
                                                                    newTrackMatchers[i].targetFrame->getCamera().getProjectionViewMatrix().dropRow(2)));

        newTrackMatchers[i].targetFrame->getSiftDescriptorDB().findMatchesConstrainedAsync(&newTrackMatchers[i], asyncMatching, matchingConfig.matchingMultithreadingBatchSize);
        numQuerries += newTrackMatchers[i].descriptors.size();
    }

    asyncMatching.waitFor();
    descriptorMatchingStats.finishedMatching(DescriptorMatchingStats::MATCH_EPIPOLAR, numQuerries);

    for (unsigned i = 0; i < newTrackMatchers.size(); i++) {
        if (newTrackMatchers[i].descriptors.empty())
            continue;


        for (unsigned j = 0; j < newTrackMatchers[i].descriptors.size(); j++) {
            const float score = newTrackMatchers[i].bestMatches[j].first;

            const unsigned sourceFeaturePointIndex = newTrackMatchers[i].descriptors[j]->userData;
            const unsigned destinationDescriptorIndex = newTrackMatchers[i].bestMatches[j].second;
            const SiftDescriptorDB::SiftDescriptor &targetDescriptor = newTrackMatchers[i].targetFrame->getSiftDescriptorDB().getDescriptor(destinationDescriptorIndex);
            unsigned destinationFeaturePointIndex = targetDescriptor.userData;

            m_interFrameRelations[i].markFeaturePointPairAsTried(sourceFeaturePointIndex, destinationFeaturePointIndex);

            if (score >= matchingConfig.frameNewTracksMatchingConfig.minScoreThreshold) {
                NewTrackCandidate candidate;
                candidate.frame[0] = this;
                candidate.frame[1] = newTrackMatchers[i].targetFrame;
                candidate.featurePointIndex[0] = sourceFeaturePointIndex;
                candidate.featurePointIndex[1] = destinationFeaturePointIndex;

                newTrackCandidates.push_back(candidate);
            } else {
                m_interFrameRelations[i].markFeaturePointAsDry(sourceFeaturePointIndex);
            }
        }
    }

    if (matchingConfig.verbatimLevel >= 2)
        std::cout << "numTrack candidates " << newTrackCandidates.size() << std::endl;
}

void Frame::dumpGridCoverage(const std::string &filename)
{
    RasterImage image;
    image.loadFromFile(m_imageFilename.c_str());

    unsigned gridSubdivs = m_observationGrid.getNumSubdivs();

    float gridScale = image.getWidth() / (gridSubdivs-1);
    float gridVerticalOffset = ((int)image.getHeight() - (int)image.getWidth()) * 0.5f;

    for (unsigned j = 0; j < gridSubdivs; j++) {
        for (unsigned i = 0; i < gridSubdivs; i++) {
            int x1 = (i+0)*gridScale;
            int x2 = (i+1)*gridScale;

            int y1 = (j+0)*gridScale + gridVerticalOffset;
            int y2 = (j+1)*gridScale + gridVerticalOffset;

            uint32_t color = (std::min<unsigned>(255, m_observationGrid.getCell(i, j).patches.size() * 32) << 0) |
                             (std::min<unsigned>(255, m_observationGrid.getCell(i, j).observations.size() * 32) << 8) |
                             (0xFF << 24);

            image.drawBox(LinAlg::Fill<int>(x1, y1),
                          LinAlg::Fill<int>(x2-1, y2-1), color, false);
        }
    }
    image.writeToFile(filename.c_str());
}

void Frame::voteForBASubset(const config::SubsetSelectionConfig &subsetSelectionConfig)
{
    unsigned gridSubdivs = m_observationGrid.getNumSubdivs();

    std::vector<std::pair<float, TrackObservation*> > scores;
    for (unsigned j = 0; j < gridSubdivs; j+=subsetSelectionConfig.numCells) {
        for (unsigned i = 0; i < gridSubdivs; i+=subsetSelectionConfig.numCells) {
            scores.clear();

            for (unsigned k = 0; k < subsetSelectionConfig.numCells; k++) {
                for (unsigned l = 0; l < subsetSelectionConfig.numCells; l++) {
                    ObservationGrid::Cell &cell = m_observationGrid.getCell(i+k, j+l);

                    if (cell.observations.size() == 0)
                        continue;

                    scores.reserve(scores.size()+cell.observations.size());

                    for (LinkedListEntry<TrackObservation> *it = cell.observations.getFirst(); it != NULL; it = it->getNext()) {
                        if (it->getHost().isFaulty())
                            continue;

                        unsigned numObs = 0;
                        Track &track = *it->getHost().getTrack();
                        for (TrackObservation &observation : track.getObservations()) {
                            if (!observation.isFaulty()) {
                                numObs++;
                            }
                        }
                        float score = numObs;

                        score = numObs * it->getHost().getWeight();

                        float sqrScreenSpaceError = it->getHost().computeSQRScreenSpaceError();
                        score *= 1.0f / (1.0f + sqrScreenSpaceError * subsetSelectionConfig.scoreSqrScreenSpaceErrorFactor);

                        scores.push_back(std::pair<float, TrackObservation*>(score, &it->getHost()));
                    }
                }
            }
            std::sort(scores.begin(), scores.end());
            for (unsigned i = 0; (i < scores.size()) && (i < subsetSelectionConfig.numVotesPerRegion); i++) {
                scores[scores.size()-1-i].second->getTrack()->addBASubsetVote();
            }
        }
    }
}


}
