/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef LENSDISTORTIONGRID_H
#define LENSDISTORTIONGRID_H

#include "../tools/LinAlg.h"
#include <string>

namespace PSFM {

class LensDistortionGrid
{
    public:
        struct DistortionObs {
            LinAlg::Vector2f distortionVector;
            float weight;
        };

        struct Cell {
            std::vector<DistortionObs> obs;
        };

        LensDistortionGrid(unsigned subdivs);
        ~LensDistortionGrid();

        void clear();

        inline void getCellCoords(const LinAlg::Vector2f &screenSpacePosition, unsigned &x, unsigned &y) const {
            int ix = (screenSpacePosition[0] * 0.5f+0.5f)*(m_subdivs-1);
            int iy = (screenSpacePosition[1] * 0.5f+0.5f)*(m_subdivs-1);

            x = std::min<int>(std::max<int>(ix, 0), m_subdivs-1);
            y = std::min<int>(std::max<int>(iy, 0), m_subdivs-1);
        }


        inline Cell &getCell(unsigned x, unsigned y) {
            return m_cells[x+y*m_subdivs];
        }
        inline Cell &getCell(const LinAlg::Vector2f &screenSpacePosition) {
            unsigned x, y;
            getCellCoords(screenSpacePosition, x, y);
            return getCell(x, y);
        }


        inline const Cell &getCell(unsigned x, unsigned y) const {
            return m_cells[x+y*m_subdivs];
        }
        inline const Cell &getCell(const LinAlg::Vector2f &screenSpacePosition) const {
            unsigned x, y;
            getCellCoords(screenSpacePosition, x, y);
            return getCell(x, y);
        }

        inline unsigned getNumSubdivs() const { return m_subdivs; }


        void debugDraw(const std::string &filename, const std::string &backgroundImage) const;
    protected:
        unsigned m_subdivs;
        std::vector<Cell> m_cells;
};

}

#endif // LENSDISTORTIONGRID_H
