/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "SFM.h"

#include "Frame.h"
#include "InternalCameraCalibration.h"

#include "../cudaInterface/ImageMipchainBuilder.h"
#include "../cudaInterface/CudaSift.h"
#include "../cudaInterface/InitialPatchOptimizer.h"
#include "../cudaInterface/SourceImageToPatchAtlasTransfer.h"
#include "../cudaInterface/trackAlignmentOptimizer.h"
#include "../cudaInterface/NewTrackObservationTester.h"
#include "../cudaInterface/PairwisePatchAligner.h"
#include "../cudaInterface/PackedPatchToPatchCacheTransfer.h"
#include "../tools/PCVToolbox.hpp"
#include "../tools/CPUStopWatch.h"
#include "../tools/TaskScheduler.h"
#include "BifocalRANSACFilter.h"
#include "ViewMatrixEstimator.h"
#include "../cudaUtilities/cudaProfilingScope.h"
#include "DuplicateTrackRemoval.h"

#include "BundleAdjustment/SingleTrackBundleAdjustment.h"
#include "Track.h"
#include "TrackObservation.h"
#include "Frame.h"

#include <algorithm>
#include <set>
#include <map>
#include <boost/optional.hpp>
#include <fstream>






namespace SFM {

const char *SFM::PerformanceStatistics::TimerToStr(Timer timer)
{
    switch (timer) {
        case TIMER_BUNDLE_ADJUSTMENT: return "Bundle adjustment";
        case TIMER_ADJUST_SEMI_ACTIVE: return "Adjust semi active tracks";
        case TIMER_TRACK_ALIGNMENT: return "Track alignment";
        case TIMER_NEW_TRACK_GATHERING: return "New track gathering";
        case TIMER_NEW_TRACK_TESTING: return "New track testing";
        case TIMER_NEW_OBS_GATHERING: return "New observation gathering";
        case TIMER_NEW_OBS_TESTING: return "New observation testing";
        case TIMER_NEW_FRAME_PHASE1: return "Adding new frames phase 1";
        case TIMER_NEW_FRAME_PHASE2_GATHER: return "Adding new frames phase 2 gather";
        case TIMER_NEW_FRAME_PHASE2_TEST: return "Adding new frames phase 2 test";
        case TIMER_NEW_FRAME_PHASE2_RANSAC: return "Adding new frames phase 2 ransac";
        case TIMER_FEATURE_EXTRACTION: return "Feature extraction";
        case TIMER_REMOVE_DUPLICATES: return "Removing duplicate tracks and observations";
        default: return "invalid enum";
    }
}

SFM::PerformanceStatistics::PerformanceStatistics()
{
    m_overallStopWatch.start();
    memset(&m_currentTimeFrame, 0, sizeof(m_currentTimeFrame));
    m_currentTimeFrame.m_interval[0] = m_overallStopWatch.getNanoseconds();
}

void SFM::PerformanceStatistics::endFrame()
{
    m_currentTimeFrame.m_interval[1] = m_overallStopWatch.getNanoseconds();
    m_timeFrames.push_back(m_currentTimeFrame);
}

void SFM::PerformanceStatistics::startNewFrame()
{
    endFrame();
    memset(&m_currentTimeFrame, 0, sizeof(m_currentTimeFrame));
    m_currentTimeFrame.m_interval[0] = m_overallStopWatch.getNanoseconds();
}

void SFM::PerformanceStatistics::startTimer(Timer timer)
{
    m_subTimer[timer].start();
}

void SFM::PerformanceStatistics::stopTimer(Timer timer)
{
    m_currentTimeFrame.m_timerSum[timer] += m_subTimer[timer].getNanoseconds();
}

void SFM::PerformanceStatistics::dump()
{
    endFrame();

    std::cout << "t = [";
    for (unsigned i = 0; i < m_timeFrames.size(); i++) {
        std::cout << " " << m_timeFrames[i].m_interval[1] * 1e-9f;
    }
    std::cout << "]';" << std::endl;

    std::cout << "y = [";
    for (unsigned i = 0; i < m_timeFrames.size(); i++) {
        uint64_t unaccounted = m_timeFrames[i].m_interval[1] - m_timeFrames[i].m_interval[0];

        for (unsigned j = 0; j < NUM_TIMERS; j++) {
            unaccounted -= m_timeFrames[i].m_timerSum[j];
            std::cout << " " << m_timeFrames[i].m_timerSum[j] * 1e-9f;
        }
        std::cout << " " << unaccounted * 1e-9f << "; ..." << std::endl;
    }
    std::cout << "]';" << std::endl;
    std::cout << "figure(1);" << std::endl;
    std::cout << "area(t, y');" << std::endl;
    //std::cout << "legend('Bundle adjustment', 'unaccounted');" << std::endl;
    std::cout << "legend(";
    for (unsigned j = 0; j < NUM_TIMERS; j++) {
        if (j > 0)
            std::cout << ", ";
        std::cout << "'"<<TimerToStr((Timer)j)<< "'";
    }
    std::cout << ", 'unaccounted');" << std::endl;

    std::cout << "figure(2);" << std::endl;
    std::cout << "pie(sum(y'));" << std::endl;
    //std::cout << "legend('Bundle adjustment', 'unaccounted');" << std::endl;
    std::cout << "legend(";
    for (unsigned j = 0; j < NUM_TIMERS; j++) {
        if (j > 0)
            std::cout << ", ";
        std::cout << "'"<<TimerToStr((Timer)j)<< "'";
    }
    std::cout << ", 'unaccounted');" << std::endl;
}


DescriptorMatchingStats::DescriptorMatchingStats()
{
    for (unsigned i = 0; i < MATCH_MODE_COUNT; i++) {
        m_numMatchingOperations[i] = 0;
        m_nanosecondsSpendMatching[i] = 0;
    }
    m_currentlyMatching = false;
}

void DescriptorMatchingStats::startMatching()
{
    assert(!m_currentlyMatching);
    m_currentlyMatching = true;
    m_stopWatch.start();
}

void DescriptorMatchingStats::finishedMatching(MatchMode mode, uint64_t numMatchingOps)
{
    assert(m_currentlyMatching);
    m_currentlyMatching = false;
    m_nanosecondsSpendMatching[mode] += m_stopWatch.getNanoseconds();
    m_numMatchingOperations[mode] += numMatchingOps;
}




SFM::SFM(const config::SfMConfig &config) : m_config(config), m_bundleAdjustment(m_config.bundleAdjustmentStructureConfig)
{
    m_bundleAdjustment.changeParameterConfig(m_config.bundleAdjustmentParameterConfig);
    m_timestampClock = 0;
    m_numCyclesWithoutNewImages = 0;
}

SFM::~SFM()
{
    clear();
}

void SFM::clear()
{
    m_trackList.clear();
    m_frames.clear();
    m_internalCameraCalibrations.clear();

    m_bundleAdjustment.clear();

    m_timestampClock = 0;
    m_numCyclesWithoutNewImages = 0;
}


unsigned SFM::testbenchAddCalibration(const LinAlg::Matrix4x4f &priorProjectionMatrix,
                                       const BundleAdjustment::RadialDistortionParametrization &radialDistortion,
                                       float aspectRatio)
{
    InternalCameraCalibration *calibration = new InternalCameraCalibration(&m_bundleAdjustment, priorProjectionMatrix, radialDistortion, aspectRatio);
    m_internalCameraCalibrations.push_back(std::unique_ptr<InternalCameraCalibration>(calibration));
    calibration->activateBA();
    return m_internalCameraCalibrations.size()-1;
}

unsigned SFM::testbenchAddFrame(unsigned calibIndex, std::vector<CudaSift::FeaturePoint> &featurePoints, unsigned width, unsigned height, const LinAlg::Matrix4x4f &viewMatrix)
{
    Frame *frame = new Frame(m_internalCameraCalibrations[calibIndex].get(), "unnamed", m_frames.size());
    m_frames.push_back(std::unique_ptr<Frame>(frame));

    frame->setInitialFeaturePoints(&featurePoints[0], featurePoints.size(), width, height, true);
    frame->getCamera().setViewMatrix(viewMatrix);

    return m_frames.size()-1;
}

unsigned SFM::testbenchAddTrack(const LinAlg::Vector4f &wsPos)
{
    const unsigned trackIndex = m_trackList.allocate();
    m_trackList[trackIndex].setup(&m_bundleAdjustment);
    m_trackList[trackIndex].setWSPosition(wsPos);

    return trackIndex;
}

unsigned SFM::testbenchAddTrackObservation(unsigned trackIndex, unsigned cameraIndex, unsigned featurePointIndex)
{
    return m_trackList[trackIndex].addObservation(m_frames[cameraIndex].get(), featurePointIndex);
}

void SFM::testbenchMakeAllTracksActive()
{
    /*
    for (unsigned i = 0; i < m_trackList.reservedSize(); i++) {
        if (!m_trackList.inUse(i)) continue;
        m_trackList[i].switchState(Track::STATE_ACTIVE);
    }
    */
    for (Track &track : m_trackList)
        track.switchState(Track::STATE_ACTIVE);
}
void SFM::testbenchMakeAllFramesActive()
{
    for (unsigned i = 0; i < m_frames.size(); i++) {
        m_frames[i]->activate(m_trackList, m_frames[i]->getCamera().getViewMatrix(), m_bundleAdjustment);
    }
}



void SFM::initSingleCameraPath(const LinAlg::Matrix4x4f &priorProjectionMatrix,
                                const BundleAdjustment::RadialDistortionParametrization &radialDistortion,
                                float aspectRatio,
                                const std::vector<std::string> &filenames)
{
    clear();

    InternalCameraCalibration *calibration = new InternalCameraCalibration(&m_bundleAdjustment, priorProjectionMatrix, radialDistortion, aspectRatio);
    m_internalCameraCalibrations.push_back(std::unique_ptr<InternalCameraCalibration>(calibration));

    m_frames.resize(filenames.size());
    for (unsigned i = 0; i < m_frames.size(); i++) {
        m_frames[i] = std::unique_ptr<Frame>(new Frame(calibration, filenames[i], i));
    }
}

void SFM::extractFeaturePoints(const std::function<void(const ExtractFeaturePointsProgressReport&)> &progressCallback)
{
    m_performanceStatistics.startTimer(PerformanceStatistics::TIMER_FEATURE_EXTRACTION);
    CudaSift cudaSift(m_config.featureExtractionConfig, m_config.cudaConfig);

    ExtractFeaturePointsProgressReport progressReport;
    progressReport.totalImageNumber = m_frames.size();
    progressReport.numfeaturePointsSoFar = 0;

    const unsigned RingBufferSize = 2;
    RasterImage srcImage[RingBufferSize];
    std::vector<CudaSift::FeaturePoint> candidates[RingBufferSize];
    unsigned ringBufferIndex = 0;

    Task loadTask;
    Task addToFrameTask;

    if (m_frames.size() > 0) {
        loadTask.schedule(boost::bind(&RasterImage::loadFromFile,
                                      &srcImage[ringBufferIndex % RingBufferSize],
                                      m_frames[0]->getImageFilename().c_str()),
                        TaskScheduler::get());
    }

    for (unsigned i = 0; i < m_frames.size(); i++) {
        //std::cout << "Processing frame " << i << std::endl;
        {
            AddCudaScopedProfileInterval("Waiting for image from disc");
            //srcImage.loadFromFile(m_frames[i]->getImageFilename().c_str());
            TaskScheduler::get().waitFor(&loadTask);
            if (i+1 < m_frames.size()) {
                loadTask.schedule(boost::bind(&RasterImage::loadFromFile,
                                              &srcImage[(ringBufferIndex+1) % RingBufferSize],
                                              m_frames[i+1]->getImageFilename().c_str()),
                                TaskScheduler::get());
            }
        }


        cudaSift.gatherFeaturePoints(srcImage[ringBufferIndex % RingBufferSize], candidates[ringBufferIndex % RingBufferSize]);

        progressReport.imageNumber = i;
        progressReport.imageFilename = m_frames[i]->getImageFilename();
        progressReport.numfeaturePointsSoFar += candidates[ringBufferIndex % RingBufferSize].size();
        if (progressCallback)
            progressCallback(progressReport);

        {
            AddCudaScopedProfileInterval("Waiting for: Adding feature points to frame");
#if 0
            std::sort(candidates.begin(), candidates.end());
            m_frames[i]->setInitialPatches(&candidates[0], std::min<unsigned>(50000u, candidates.size()), srcImage.getWidth());

#else
            //m_frames[i]->setInitialPatches(&candidates[0], candidates.size(), srcImage.getWidth());
            if (i > 0)
                TaskScheduler::get().waitFor(&addToFrameTask);

            addToFrameTask.schedule(boost::bind(&Frame::setInitialFeaturePoints,
                                      m_frames[i].get(),
                                      &candidates[ringBufferIndex % RingBufferSize][0],
                                      candidates[ringBufferIndex % RingBufferSize].size(),
                                      srcImage[ringBufferIndex % RingBufferSize].getWidth(),
                                      srcImage[ringBufferIndex % RingBufferSize].getHeight(),
                                      m_frames.size(), false),
                        TaskScheduler::get());
#endif
        }
        ringBufferIndex++;
    }

    if (m_frames.size() > 0)
        TaskScheduler::get().waitFor(&addToFrameTask);

    m_performanceStatistics.stopTimer(PerformanceStatistics::TIMER_FEATURE_EXTRACTION);

/*
    unsigned numPairwiseDBLookups = 0;
    for (unsigned i = 0; i < m_frames.size(); i++) {
        for (unsigned j = i+1; j < m_frames.size(); j++) {
            numPairwiseDBLookups += m_frames[j]->getSiftDescriptorDB().getNumDescriptors();
        }
    }

    std::cout << "Num theoretical pairwise patch DB lookups: " << numPairwiseDBLookups << std::endl;
*/
#if 0
    std::vector<SiftDescriptorDB::Match> matches;
    Engine::CPUStopWatch stopWatch;
    for (unsigned i = 0; i < m_frames.size(); i++)
        for (unsigned j = i+1; j < m_frames.size(); j++) {
            matches.clear();
            m_frames[i]->getSiftDescriptorDB().findMatches(m_frames[j]->getSiftDescriptorDB(), matches);
            std::cout << "at " << stopWatch.getNanoseconds()  * 1e-6 << " ms: " << i << " vs " << j << " done" << std::endl;
        }

    std::cout << "Time spend on pairwise patch DB lookups: " << stopWatch.getNanoseconds() << " nanoseconds" << std::endl;
#endif
}

void SFM::buildInitialPose(unsigned frame1, unsigned frame2)
{
    AddCudaScopedProfileInterval("Building initial pose");

    if (m_config.verbatimLevel >= 1)
        std::cout << "Building initial pose!" << std::endl;

    InternalCameraCalibration *calib1 = m_frames[frame1]->getCamera().getInternalCalibration();
    InternalCameraCalibration *calib2 = m_frames[frame2]->getCamera().getInternalCalibration();

    if (calib1->getBAHandle() == (unsigned)-1)
        calib1->activateBA();

    if (calib2->getBAHandle() == (unsigned)-1)
        calib2->activateBA();

    m_frames[frame1]->getCamera().activateBA(m_bundleAdjustment);
    m_frames[frame2]->getCamera().activateBA(m_bundleAdjustment);

    LinAlg::Matrix3x3f K1;
    K1[0] = calib1->getProjectionMatrix()[0].StripHom();
    K1[1] = calib1->getProjectionMatrix()[1].StripHom();
    K1[2] = calib1->getProjectionMatrix()[3].StripHom();

    LinAlg::Matrix3x3f KInv1 = K1;
    KInv1.GaussJordanInvert();

    LinAlg::Matrix3x3f K2;
    K2[0] = calib2->getProjectionMatrix()[0].StripHom();
    K2[1] = calib2->getProjectionMatrix()[1].StripHom();
    K2[2] = calib2->getProjectionMatrix()[3].StripHom();

    LinAlg::Matrix3x3f KInv2 = K2;
    KInv2.GaussJordanInvert();


    std::vector<SiftDescriptorDB::Match> matches;
    m_descriptorMatchingStats.startMatching();
    {
        AddCudaScopedProfileInterval("Key matching");
        m_frames[frame1]->getSiftDescriptorDB().findMatches(m_frames[frame2]->getSiftDescriptorDB(), matches,
                                            m_config.featureMatchingConfig.matchingMultithreadingBatchSize);
    }

    m_descriptorMatchingStats.finishedMatching(DescriptorMatchingStats::MATCH_INITIAL, m_frames[frame2]->getSiftDescriptorDB().getNumDescriptors());

    if (m_config.verbatimLevel >= 2)
        std::cout << "found " << matches.size() << " matches!" << std::endl;

    BifocalRANSACFilter bifocalFilter;
    std::vector<BifocalRANSACFilter::Match> filterMatches;
    filterMatches.resize(matches.size());
    for (unsigned i = 0; i < matches.size(); i++) {
        if (matches[i].closestMatchIndex == (unsigned) -1) {
            filterMatches.resize(i);
            break;
        }
        if (matches[i].normalizedDiff > m_config.featureMatchingConfig.initialImagePairMatchingThreshold) {
            filterMatches.resize(i);
            break;
        }

        filterMatches[i].index[0] = m_frames[frame1]->getSiftDescriptorDB().getDescriptor(matches[i].closestMatchIndex).userData;
        filterMatches[i].index[1] = m_frames[frame2]->getSiftDescriptorDB().getDescriptor(matches[i].srcIndex).userData;


        const Frame::ImageFeaturePoint &FP1 = m_frames[frame1]->getFeaturePoint(filterMatches[i].index[0]);
        const Frame::ImageFeaturePoint &FP2 = m_frames[frame2]->getFeaturePoint(filterMatches[i].index[1]);

        filterMatches[i].pos[0] = LinAlg::Fill(FP1.x * 2000, FP1.y * 2000);
        filterMatches[i].pos[1] = LinAlg::Fill(FP2.x * 2000, FP2.y * 2000);

        filterMatches[i].size[0] = FP1.size * 2000;
        filterMatches[i].size[1] = FP2.size * 2000;

        filterMatches[i].angle[0] = FP1.angle;
        filterMatches[i].angle[1] = FP2.angle;

        filterMatches[i].normalizedDifference = matches[i].normalizedDiff;
    }

    if (m_config.verbatimLevel >= 2)
        std::cout << "left for ransac: " << filterMatches.size() << " matches!" << std::endl;

    LinAlg::Matrix3x3f bestFundamental;
    std::vector<float> matchProbs;
    matchProbs.resize(filterMatches.size());
    {
        AddCudaScopedProfileInterval("RANSAC");
        bifocalFilter.filter(&filterMatches[0], &matchProbs[0], filterMatches.size(), 50000, bestFundamental, 1.0f);
    }


    for (unsigned i = 0; i < filterMatches.size(); ) {
        if (matchProbs[i] < 0.1f) {
            matchProbs[i] = matchProbs[matchProbs.size()-1];
            filterMatches[i] = filterMatches[filterMatches.size()-1];
            matchProbs.resize(matchProbs.size()-1);
            filterMatches.resize(filterMatches.size()-1);
        } else i++;
    }
    if (m_config.verbatimLevel >= 2)
        std::cout << "Remaining: " << filterMatches.size() << std::endl;

    std::vector<LinAlg::Vector3f> normalizedPoints1;
    std::vector<LinAlg::Vector3f> normalizedPoints2;

    normalizedPoints1.reserve(filterMatches.size());
    normalizedPoints2.reserve(filterMatches.size());
    for (unsigned i = 0; i < filterMatches.size(); i++) {
        const Frame::ImageFeaturePoint &FP1 = m_frames[frame1]->getFeaturePoint(filterMatches[i].index[0]);
        const Frame::ImageFeaturePoint &FP2 = m_frames[frame2]->getFeaturePoint(filterMatches[i].index[1]);
        {
            LinAlg::Vector3f P = KInv1 * LinAlg::Fill(FP1.x, FP1.y, 1.0f);
            P /= P[2];
            normalizedPoints1.push_back(P);
        }

        {
            LinAlg::Vector3f P = KInv2 * LinAlg::Fill(FP2.x, FP2.y, 1.0f);
            P /= P[2];
            normalizedPoints2.push_back(P);
        }
    }


    LinAlg::Matrix3x4f P1;

    LinAlg::Matrix3x3f essential = PCV::getFundamentalMatrix(normalizedPoints1, normalizedPoints2);

    unsigned bestNumPositive = 0;
    unsigned bestVariation = 0;
    for (unsigned variation = 0; variation < 4; variation++) {
        LinAlg::Matrix3x4f P2 = PCV::buildProjectionMatrix(essential, 0.1f, variation);
        unsigned numPositive = 0;
        for (unsigned i = 0; i < normalizedPoints1.size(); i++) {
            LinAlg::Vector<4, float> p = PCV::triangulatePoint(P1, normalizedPoints1[i], P2, normalizedPoints2[i]);

            LinAlg::Vector<3, float> U = P1 * p;
            LinAlg::Vector<3, float> V = P2 * p;

            if ((U[2] > 0.0) && (V[2] > 0.0))
                numPositive++;
        }
        if (numPositive > bestNumPositive) {
            bestNumPositive = numPositive;
            bestVariation = variation;
        }
    }
    LinAlg::Matrix<3, 4, float> P2 = PCV::buildProjectionMatrix(essential, 0.1f, bestVariation);

#if 1
    {
        LinAlg::Matrix4x4f M;
        M[0] = P1[0];
        M[1] = P1[1];
        M[3] = P1[2];

        M = M * LinAlg::Translation3D(LinAlg::Fill(0.0f, 0.0f, 0.0f)) * LinAlg::RotateZ((float)M_PI);
        P1[0] = M[0];
        P1[1] = M[1];
        P1[2] = M[3];
    }
    {
        LinAlg::Matrix4x4f M;
        M[0] = P2[0];
        M[1] = P2[1];
        M[3] = P2[2];

        M = M * LinAlg::Translation3D(LinAlg::Fill(0.0f, 0.0f, 0.0f)) * LinAlg::RotateZ((float)M_PI);
        P2[0] = M[0];
        P2[1] = M[1];
        P2[2] = M[3];
    }
#elif 0
    {
        LinAlg::Matrix4x4f M;
        M[0] = P1[0];
        M[1] = P1[1];
        M[3] = P1[2];

        M = M * LinAlg::Translation3D(LinAlg::Fill(0.0f, 0.0f, 0.0f)) * LinAlg::RotateY((float)M_PI);
        P1[0] = M[0];
        P1[1] = M[1];
        P1[2] = M[3];
    }
    {
        LinAlg::Matrix4x4f M;
        M[0] = P2[0];
        M[1] = P2[1];
        M[3] = P2[2];

        M = M * LinAlg::Translation3D(LinAlg::Fill(0.0f, 0.0f, 0.0f)) * LinAlg::RotateY((float)M_PI);
        P2[0] = M[0];
        P2[1] = M[1];
        P2[2] = M[3];
    }
#endif

#if 1
    for (unsigned i = 0; i < filterMatches.size(); i++) {
/*
        const Frame::ImagePatch &patch1 = m_frames[frame1]->getFeaturePoint(filterMatches[i].index[0]);
        const Frame::ImagePatch &patch2 = m_frames[frame2]->getFeaturePoint(filterMatches[i].index[1]);
*/
        LinAlg::Vector4f worldSpacePosition;
        {
            worldSpacePosition = PCV::triangulatePoint(P1, normalizedPoints1[i], P2, normalizedPoints2[i]);
            float len = std::sqrt(worldSpacePosition.SQRLen());
            if (len < 1e-20f)
                continue;
            worldSpacePosition *= 1.0f / len;
/*
            worldSpacePosition[0] += rand() / (float)RAND_MAX * 0.02f - 0.01f;
            worldSpacePosition[1] += rand() / (float)RAND_MAX * 0.02f - 0.01f;
            worldSpacePosition[2] += rand() / (float)RAND_MAX * 0.02f - 0.01f;
*/
        }
        const unsigned trackIndex = m_trackList.allocate();
        m_trackList[trackIndex].setup(&m_bundleAdjustment);
        m_trackList[trackIndex].setWSPosition(worldSpacePosition);
        unsigned obsIndex1 = m_trackList[trackIndex].addObservation(m_frames[frame1].get(), filterMatches[i].index[0]);
        m_trackList[trackIndex].addObservation(m_frames[frame2].get(), filterMatches[i].index[1]);

        m_trackList[trackIndex].setReferenceObservation(&m_trackList[trackIndex].getObservations()[obsIndex1]);

        m_trackList[trackIndex].switchState(Track::STATE_ACTIVE);
        m_trackList[trackIndex].updateTrackMatchingDescriptor(nullptr);
    }
#endif

    LinAlg::Matrix3x3f dummy3;
    LinAlg::Matrix4x4f dummy4;

    LinAlg::Matrix4x4f worldToEye;
    LinAlg::Vector3f vecDummy;
    PCV::decomposeProjectionMatrix(P1,
                              dummy3,
                              worldToEye,
                              dummy4,
                              vecDummy);

    m_frames[frame1]->getCamera().setViewMatrix(worldToEye);
    m_frames[frame1]->getCamera().updateBA(m_bundleAdjustment);


    PCV::decomposeProjectionMatrix(P2,
                              dummy3,
                              worldToEye,
                              dummy4,
                              vecDummy);

/*
worldToEye[0][3] += rand() / (float)RAND_MAX * 0.02f - 0.01f;
worldToEye[1][3] += rand() / (float)RAND_MAX * 0.02f - 0.01f;
worldToEye[2][3] += rand() / (float)RAND_MAX * 0.02f - 0.01f;
*/

    m_frames[frame2]->getCamera().setViewMatrix(worldToEye);
    m_frames[frame2]->getCamera().updateBA(m_bundleAdjustment);

    m_frames[frame1]->setActive();
    m_frames[frame2]->setActive();

    m_state = OP_STATE_ADDPATCHES;

    recomputeTrackWeights();
    selectBASubset();

    m_allViewsReconstructedExtraRounds = 0;
}


void SFM::iterateBA(unsigned iterations, bool runSemiActive)
{
//Engine::CPUStopWatch stopwatch;

#if 1
    if (m_bundleAdjustment.converged())
        m_bundleAdjustment.restart();
#endif
    m_performanceStatistics.startTimer(PerformanceStatistics::TIMER_BUNDLE_ADJUSTMENT);
    {
        AddCudaScopedProfileInterval("BA");
//stopwatch.start();
        m_bundleAdjustment.iterate(iterations);
//std::cout << "BA: " << stopwatch.getNanoseconds() * 1e-9f << std::endl;
    }
//stopwatch.start();
    readbackBAData();
//std::cout << "readback BA: " << stopwatch.getNanoseconds() * 1e-9f << std::endl;
    m_performanceStatistics.stopTimer(PerformanceStatistics::TIMER_BUNDLE_ADJUSTMENT);
    if (runSemiActive) {
        m_performanceStatistics.startTimer(PerformanceStatistics::TIMER_ADJUST_SEMI_ACTIVE);
        {
            AddCudaScopedProfileInterval("readjustSemiActiveTracks");
//stopwatch.start();
            readjustSemiActiveTracks();
//std::cout << "run semiactive: " << stopwatch.getNanoseconds() * 1e-9f << std::endl;
        }
        m_performanceStatistics.stopTimer(PerformanceStatistics::TIMER_ADJUST_SEMI_ACTIVE);
    }
}

void SFM::mergeDuplicateTracks()
{
    AddCudaScopedProfileInterval("SFM::mergeDuplicateTracks()");
    m_performanceStatistics.startTimer(PerformanceStatistics::TIMER_REMOVE_DUPLICATES);

    unsigned numTracksRemoved = 0;
    {
        std::vector<uint8_t> flags;
        flags.resize(m_trackList.reservedSize());
        memset(&flags[0], 0, flags.size());

        DuplicateTrackRemoval duplTrackRem;
        duplTrackRem.clear();

        for (unsigned i = 0; i < m_trackList.reservedSize(); i++) {
            if (!m_trackList.inUse(i))
                continue;

            Track &track = m_trackList[i];

            if (track.getState() == Track::STATE_DISABLED)
                continue;

            if (std::abs(track.getLastWSPositionEstimate()[3]) < 1e-20f)
                continue;

            DuplicateTrackRemoval::Element element;
            element.m_position = track.getLastWSPositionEstimate().StripHom() / track.getLastWSPositionEstimate()[3];
            element.m_size = track.getSize();
            element.m_userData = i;

            DuplicateTrackRemoval::Element *mergeTarget = duplTrackRem.mergeOrAdd(element);

            if (mergeTarget != nullptr) {
                numTracksRemoved++;
                flags[mergeTarget->m_userData] |= 1;

                if (track.getState() == Track::STATE_ACTIVE)
                    flags[mergeTarget->m_userData] |= 2;

                Track &targetTrack = m_trackList[mergeTarget->m_userData];

                for (unsigned k = 0; k < track.getObservations().reservedSize(); k++) {
                    if (track.getObservations().inUse(k)) {
                        unsigned obsIndex = targetTrack.addObservation(track.getObservations()[k].getFrame(), track.getObservations()[k].getFrameFeaturePointIndex());

                        if (track.getObservations()[k].isFaulty())
                            targetTrack.getObservations()[obsIndex].markFaulty();
                        else
                            if (targetTrack.getState() == Track::STATE_ACTIVE)
                                targetTrack.getObservations()[obsIndex].activateBA();
                    }
                }

                track.switchState(Track::STATE_DISABLED);
                m_trackList.free(i);
            }
        }
        for (unsigned i = 0; i < m_trackList.reservedSize(); i++) {
            if (flags[i] & 1) {
                Track &track = m_trackList[i];
                track.updateTrackMatchingDescriptor(NULL);
/*
                if (flags[i] & 2)
                    track.switchState(Track::STATE_ACTIVE);
*/
            }
        }
    }

    if (m_config.verbatimLevel >= 2)
        std::cout << "Merged " << numTracksRemoved << " duplicate tracks!" << std::endl;


    unsigned numObsRemoved = 0;
    for (unsigned i = 0; i < m_trackList.reservedSize(); i++) {
        if (!m_trackList.inUse(i))
            continue;

        Track &track = m_trackList[i];

        if (track.getState() == Track::STATE_DISABLED)
            continue;

        bool trackChanged = false;

        for (unsigned j = 0; j < track.getObservations().reservedSize(); j++) {
            if (!track.getObservations().inUse(j))
                continue;
            if (track.getObservations()[j].isFaulty())
                continue;

            const float maxDistance = track.getObservations()[j].getFrame()->getFeaturePoint(track.getObservations()[j].getFrameFeaturePointIndex()).size * 4.0f;
            assert(maxDistance > 0.0f);

            for (unsigned k = j+1; k < track.getObservations().reservedSize(); k++) {
                if (!track.getObservations().inUse(k))
                    continue;
                if (track.getObservations()[k].isFaulty())
                    continue;
                if (track.getObservations()[j].getFrame() != track.getObservations()[k].getFrame())
                    continue;



                if ((track.getObservations()[j].getFrameFeaturePointIndex() == track.getObservations()[k].getFrameFeaturePointIndex()) ||
                    ((track.getObservations()[j].getScreenSpacePosition() - track.getObservations()[k].getScreenSpacePosition()).SQRLen() < maxDistance*maxDistance)) {
                    track.getObservations()[k].markFaulty();
                    numObsRemoved++;

                    trackChanged = true;
                }
            }
        }

        if (trackChanged)
            track.updateTrackMatchingDescriptor(NULL);
    }

    if (m_config.verbatimLevel >= 2)
        std::cout << "Merged " << numObsRemoved << " duplicate observations!" << std::endl;

    m_performanceStatistics.stopTimer(PerformanceStatistics::TIMER_REMOVE_DUPLICATES);
}


void SFM::activateNextFrames()
{
    AddCudaScopedProfileInterval("New frames");

    if (m_config.verbatimLevel >= 1)
        std::cout << "Adding new frames" << std::endl;

    m_performanceStatistics.startTimer(PerformanceStatistics::TIMER_NEW_FRAME_PHASE1);

    std::vector<float> scores;

    scores.resize(m_frames.size());

    std::vector<SiftDescriptorDB::SiftDescriptor> globalPatchDescrList;
    globalPatchDescrList.reserve(m_trackList.reservedSize() * 4);

    for (unsigned i = 0; i < m_trackList.reservedSize(); i++) {
        if (m_trackList.inUse(i)) {
            Track &track = m_trackList[i];
            if (track.getState() == Track::STATE_DISABLED)
                continue;

            assert(track.getTrackMatchingDescriptor() != NULL);

            unsigned offset = globalPatchDescrList.size();
            globalPatchDescrList.resize(offset + track.getTrackMatchingDescriptor()->getDescriptors().size());
            memcpy(&globalPatchDescrList[offset], &track.getTrackMatchingDescriptor()->getDescriptors()[0], track.getTrackMatchingDescriptor()->getDescriptors().size() * sizeof(SiftDescriptorDB::SiftDescriptor));
            for (unsigned j = 0; j < track.getTrackMatchingDescriptor()->getDescriptors().size(); j++) {
                globalPatchDescrList[offset+j].userData = i;
            }
        }
    }
#if 0
    {
        std::cout << "globalPatchDescrList.size() = " << globalPatchDescrList.size() << std::endl;
        unsigned numObs = 0;
        for (unsigned i = 0; i < m_trackList.reservedSize(); i++) {
            if (!m_trackList.inUse(i))
                continue;
            Track &track = m_trackList[i];
            if (track.getState() == Track::STATE_DISABLED)
                continue;

            for (unsigned j = 0; j < track.getObservations().reservedSize(); j++) {
                if (!track.getObservations().inUse(j))
                    continue;
                const TrackObservation &obs = track.getObservations()[j];
                if (!obs.isFaulty())
                    numObs++;
            }
        }
        std::cout << "numObs = " << numObs << std::endl;
        assert(numObs == globalPatchDescrList.size());
    }
#endif

    for (unsigned i = 0; i < m_frames.size(); i++) {
        if (m_frames[i]->active()) continue;

        LinAlg::Matrix4x4f viewMatrix;
        float score;
        m_frames[i]->ransacEstimateViewMatrixFromTracks(m_trackList, viewMatrix, score, globalPatchDescrList,
                                                        m_config.featureMatchingConfig, m_descriptorMatchingStats);


        if (m_config.verbatimLevel >= 2)
            std::cout << "Score for " << i<< ": " << score << std::endl;

        m_frames[i]->getCamera().setViewMatrix(viewMatrix);

        scores[i] = score;
    }

    m_performanceStatistics.stopTimer(PerformanceStatistics::TIMER_NEW_FRAME_PHASE1);
    m_performanceStatistics.startTimer(PerformanceStatistics::TIMER_NEW_FRAME_PHASE2_GATHER);


    std::vector<Frame::NewObservationCandidate> candidates;

    for (unsigned i = 0; i < m_frames.size(); i++) {
        if (m_frames[i]->active()) continue;

        if (scores[i] > m_config.featureMatchingConfig.frameNewCameraMatchingConfig.minInitialRansacScore) {
#ifdef SFM_PERFORM_PATCH_ALIGNMENT
            m_frames[i]->findNewObservations(m_trackList, candidates, 0, m_descriptorMatchingStats, m_config.featureMatchingConfig, true);
#else
            m_frames[i]->findNewObservations(m_trackList, candidates, 0, m_descriptorMatchingStats, m_config.featureMatchingConfig, true);
#endif
        }
    }
    m_performanceStatistics.stopTimer(PerformanceStatistics::TIMER_NEW_FRAME_PHASE2_GATHER);

    m_performanceStatistics.startTimer(PerformanceStatistics::TIMER_NEW_FRAME_PHASE2_TEST);
#ifdef SFM_PERFORM_PATCH_ALIGNMENT

    std::vector<FeaturePointAlignmentEstimator::FeaturePointPair> featurePointAlignmentPairs;
    featurePointAlignmentPairs.resize(candidates.size());

    for (unsigned i = 0; i < featurePointAlignmentPairs.size(); i++) {
        Track &track = m_trackList[candidates[i].trackIndex];
        assert(track.getReferenceObservation() != NULL);
        TrackObservation &templateObs = *track.getReferenceObservation();

        featurePointAlignmentPairs[i].templateFPFrame = templateObs.getFrame();
        featurePointAlignmentPairs[i].templateFPIndex = templateObs.getFrameFeaturePointIndex();
        featurePointAlignmentPairs[i].templateFPScreenSpacePosition = templateObs.getScreenSpacePosition();


        featurePointAlignmentPairs[i].secondFPFrame = candidates[i].frame;
        featurePointAlignmentPairs[i].secondFPIndex = candidates[i].featurePointIndex;
        featurePointAlignmentPairs[i].secondFPScreenSpacePosition[0] = candidates[i].frame->getFeaturePoint(candidates[i].featurePointIndex).x;
        featurePointAlignmentPairs[i].secondFPScreenSpacePosition[1] = candidates[i].frame->getFeaturePoint(candidates[i].featurePointIndex).y;

        featurePointAlignmentPairs[i].normal = track.getNormal();
        featurePointAlignmentPairs[i].estimatedTrackPosition = track.getLastWSPositionEstimate();
    }

    m_featurePointAlignmentEstimator.process(featurePointAlignmentPairs);
#endif
    m_performanceStatistics.stopTimer(PerformanceStatistics::TIMER_NEW_FRAME_PHASE2_TEST);

    m_performanceStatistics.startTimer(PerformanceStatistics::TIMER_NEW_FRAME_PHASE2_RANSAC);

    std::vector<ViewMatrixEstimator::ControlPoint> controlPoints;
    controlPoints.reserve(5000);

    std::vector<unsigned> controlPointToFeaturePointPair;
    controlPointToFeaturePointPair.reserve(5000);


    unsigned numNewViews = 0;

    ViewMatrixEstimator estimator;

    for (unsigned i = 0; i < m_frames.size(); i++) {
        if (m_frames[i]->active()) continue;

        controlPoints.clear();
        controlPointToFeaturePointPair.clear();
        for (unsigned j = 0; j < candidates.size(); j++) {
            if ((candidates[j].frame == m_frames[i].get())
#ifdef SFM_PERFORM_PATCH_ALIGNMENT
                 && (featurePointAlignmentPairs[j].residualError < 1.5f)) {
#else
                 ) {
#endif
                ViewMatrixEstimator::ControlPoint cp;
                cp.index[0] = candidates[j].trackIndex;
                cp.index[1] = candidates[j].featurePointIndex;
                cp.worldSpacePosition = m_trackList[candidates[j].trackIndex].getLastWSPositionEstimate();
                cp.rcpSqrScreenSize = 1.0f / (m_frames[i]->getFeaturePoint(candidates[j].featurePointIndex).size * m_frames[i]->getFeaturePoint(candidates[j].featurePointIndex).size);
#ifdef SFM_PERFORM_PATCH_ALIGNMENT
                cp.normalizedDifference = featurePointAlignmentPairs[j].residualError * 0.7f;
                cp.screenSpacePosition = featurePointAlignmentPairs[j].secondFPScreenSpacePosition;
#else
                cp.normalizedDifference = candidates[j].minNormDifference;
                cp.screenSpacePosition[0] = m_frames[i]->getFeaturePoint(candidates[j].featurePointIndex).x;
                cp.screenSpacePosition[1] = m_frames[i]->getFeaturePoint(candidates[j].featurePointIndex).y;
#endif

                controlPoints.push_back(cp);
                controlPointToFeaturePointPair.push_back(j);
            }
        }
        if (m_config.verbatimLevel >= 2)
            std::cout << "Points for frame " << i << ": " << controlPoints.size() << std::endl;
        if (controlPoints.size() > m_config.featureMatchingConfig.frameNewCameraMatchingConfig.minBootstrappingMatchCount) {
            LinAlg::Matrix4x4f viewMatrix;
            float score;

            std::vector<float> probs;
            probs.resize(controlPoints.size());

            {
                AddCudaScopedProfileInterval("Adding frame phase 2 RANSAC");
                estimator.estimate(m_frames[i]->getCamera().getInternalCalibration()->getProjectionMatrix().dropRow(2),
                                   &controlPoints[0],
                                   &probs[0],
                                   controlPoints.size(),
                                   m_config.featureMatchingConfig.frameNewCameraMatchingConfig.numSecondaryRansacIterations,
                                   viewMatrix,
                                   score);
            }

            if (m_config.verbatimLevel >= 2)
                std::cout << "second row " << i << " score: " << score << "  out of " << controlPoints.size() << std::endl;
            if (score > m_config.featureMatchingConfig.frameNewCameraMatchingConfig.minSecondaryRansacScore) {

                unsigned numAcceptableObs = 0;
                for (unsigned k = 0; k < controlPoints.size(); k++) {
                    if (probs[k] > m_config.featureMatchingConfig.frameNewCameraMatchingConfig.minNewObsScore) {
                        numAcceptableObs++;
                    }
                }

                if (numAcceptableObs > m_config.featureMatchingConfig.frameNewCameraMatchingConfig.minAcceptedObs) {
                    m_frames[i]->activate(m_trackList, viewMatrix, m_bundleAdjustment);
                    numNewViews++;
                    for (unsigned k = 0; k < controlPoints.size(); k++) {

                        if (probs[k] > m_config.featureMatchingConfig.frameNewCameraMatchingConfig.minNewObsScore) {
                            Track &track = m_trackList[controlPoints[k].index[0]];

                            unsigned obsIndex = track.addObservation(m_frames[i].get(), controlPoints[k].index[1]);
                            TrackObservation &obs = track.getObservations()[obsIndex];
                            if (track.getState() == Track::STATE_ACTIVE)
                                obs.activateBA();

#ifdef SFM_PERFORM_PATCH_ALIGNMENT
                            const FeaturePointAlignmentEstimator::FeaturePointPair &FPpair = featurePointAlignmentPairs[controlPointToFeaturePointPair[k]];

                            obs.setScreenSpacePosition(FPpair.secondFPScreenSpacePosition);
                            obs.setReferenceObsScreenToThisScreen(FPpair.resultingHomography, FPpair.residualError);
                            track.setNeedsNormalUpdate();
#endif

                            track.updateTrackMatchingDescriptor(NULL);
                        }
                    }
                }
                if (m_config.verbatimLevel >= 2)
                    std::cout << "number of accepted new obs:  " << numAcceptableObs << std::endl;
            }
        }
    }

    if (numNewViews > 0)
        m_numCyclesWithoutNewImages = 0;
    else
        m_numCyclesWithoutNewImages++;

    m_performanceStatistics.stopTimer(PerformanceStatistics::TIMER_NEW_FRAME_PHASE2_RANSAC);
}


void SFM::findAdditionalObservations()
{
    checkMajorTrackChanges();

    AddCudaScopedProfileInterval("SFM::findAdditionalObservations()");
    m_performanceStatistics.startTimer(PerformanceStatistics::TIMER_NEW_OBS_GATHERING);

    std::vector<Frame::NewObservationCandidate> candidates;
    for (unsigned i = 0; i < m_frames.size(); i++) {
        if (!m_frames[i]->active()) continue;

#ifdef SFM_PERFORM_PATCH_ALIGNMENT
        m_frames[i]->findNewObservations(m_trackList, candidates, m_timestampClock, m_descriptorMatchingStats, 8.0f, 1.5f);
#else
        m_frames[i]->findNewObservations(m_trackList, candidates, m_timestampClock, m_descriptorMatchingStats, m_config.featureMatchingConfig, false);
#endif
    }
    m_performanceStatistics.stopTimer(PerformanceStatistics::TIMER_NEW_OBS_GATHERING);

    m_performanceStatistics.startTimer(PerformanceStatistics::TIMER_NEW_OBS_TESTING);
#ifdef SFM_PERFORM_PATCH_ALIGNMENT

    std::vector<FeaturePointAlignmentEstimator::FeaturePointPair> featurePointAlignmentPairs;
    featurePointAlignmentPairs.resize(candidates.size());

    for (unsigned i = 0; i < featurePointAlignmentPairs.size(); i++) {
        Track &track = m_trackList[candidates[i].trackIndex];
        assert(track.getReferenceObservation() != NULL);
        TrackObservation &templateObs = *track.getReferenceObservation();

        featurePointAlignmentPairs[i].templateFPFrame = templateObs.getFrame();
        featurePointAlignmentPairs[i].templateFPIndex = templateObs.getFrameFeaturePointIndex();
        featurePointAlignmentPairs[i].templateFPScreenSpacePosition = templateObs.getScreenSpacePosition();


        featurePointAlignmentPairs[i].secondFPFrame = candidates[i].frame;
        featurePointAlignmentPairs[i].secondFPIndex = candidates[i].featurePointIndex;
        featurePointAlignmentPairs[i].secondFPScreenSpacePosition[0] = candidates[i].frame->getFeaturePoint(candidates[i].featurePointIndex).x;
        featurePointAlignmentPairs[i].secondFPScreenSpacePosition[1] = candidates[i].frame->getFeaturePoint(candidates[i].featurePointIndex).y;

        featurePointAlignmentPairs[i].normal = track.getNormal();
        featurePointAlignmentPairs[i].estimatedTrackPosition = track.getLastWSPositionEstimate();
    }

    m_featurePointAlignmentEstimator.process(featurePointAlignmentPairs);
#endif

    unsigned numAcceptedNewObs = 0;
    for (unsigned i = 0; i < candidates.size(); i++) {
        Track &track = m_trackList[candidates[i].trackIndex];

#ifdef SFM_PERFORM_PATCH_ALIGNMENT
        const FeaturePointAlignmentEstimator::FeaturePointPair &FPpair = featurePointAlignmentPairs[i];

        const Frame::ImageFeaturePoint &iFP = candidates[i].frame->getFeaturePoint(candidates[i].featurePointIndex);

        float movement = std::sqrt((FPpair.secondFPScreenSpacePosition - LinAlg::Fill(iFP.x, iFP.y)).SQRLen());
        if ((featurePointAlignmentPairs[i].residualError < 1.5f) && (movement < 10 / 2000.0f)) {
#else
        if (candidates[i].minNormDifference < m_config.featureMatchingConfig.frameNewObservationsMatchingConfig.maxNormalizedMatchDiff) {
#endif
            unsigned obsIndex = track.addObservation(candidates[i].frame, candidates[i].featurePointIndex);
            TrackObservation &obs = track.getObservations()[obsIndex];

            if (track.getState() == Track::STATE_ACTIVE)
                obs.activateBA();
#ifdef SFM_PERFORM_PATCH_ALIGNMENT
            obs.setScreenSpacePosition(FPpair.secondFPScreenSpacePosition);
            obs.setReferenceObsScreenToThisScreen(FPpair.resultingHomography, FPpair.residualError);
            track.setNeedsNormalUpdate();
#endif
            track.updateTrackMatchingDescriptor(NULL);
            numAcceptedNewObs++;
        }
    }

    if (m_config.verbatimLevel >= 2)
        std::cout << "AcceptedNewObs " << numAcceptedNewObs << std::endl;
    m_performanceStatistics.stopTimer(PerformanceStatistics::TIMER_NEW_OBS_TESTING);


//    checkMajorTrackChanges();

    m_performanceStatistics.startTimer(PerformanceStatistics::TIMER_NEW_TRACK_GATHERING);
    std::vector<Frame::NewTrackCandidate> newTrackCandidates;

    for (unsigned i = 0; i < m_frames.size(); i++) {
        if (!m_frames[i]->active()) continue;

        m_frames[i]->findNewTracks(m_trackList, m_frames, m_bundleAdjustment, newTrackCandidates,
                                   m_config.featureMatchingConfig,
                                   m_descriptorMatchingStats);
    }
    m_performanceStatistics.stopTimer(PerformanceStatistics::TIMER_NEW_TRACK_GATHERING);

    m_performanceStatistics.startTimer(PerformanceStatistics::TIMER_NEW_TRACK_TESTING);
    std::vector<unsigned> probationaryTrackIndices;
    probationaryTrackIndices.resize(newTrackCandidates.size());

    for (unsigned i = 0; i < newTrackCandidates.size(); i++) {
        probationaryTrackIndices[i] = m_trackList.allocate();

        Track &track = m_trackList[probationaryTrackIndices[i]];
        track.setup(&m_bundleAdjustment);

        unsigned obs1 = track.addObservation(newTrackCandidates[i].frame[0], newTrackCandidates[i].featurePointIndex[0]);
        track.addObservation(newTrackCandidates[i].frame[1], newTrackCandidates[i].featurePointIndex[1]);

        Frame::ImageFeaturePoint &FP1 = newTrackCandidates[i].frame[0]->getFeaturePoint(newTrackCandidates[i].featurePointIndex[0]);
        Frame::ImageFeaturePoint &FP2 = newTrackCandidates[i].frame[1]->getFeaturePoint(newTrackCandidates[i].featurePointIndex[1]);

        track.setWSPosition(PCV::triangulatePoint(
                            newTrackCandidates[i].frame[0]->getCamera().getProjectionViewMatrix().dropRow(2),
                            LinAlg::Fill(FP1.x, FP1.y, 1.0f),
                            newTrackCandidates[i].frame[1]->getCamera().getProjectionViewMatrix().dropRow(2),
                            LinAlg::Fill(FP2.x, FP2.y, 1.0f)));

        track.setReferenceObservation(&track.getObservations()[obs1]);
        track.switchState(Track::STATE_SEMI_ACTIVE);
    }
#ifdef SFM_PERFORM_PATCH_ALIGNMENT

    computeMissingFeaturePointAlignments(false);
    updateTrackNormals();


    unsigned numTracksAccepted = 0;
    for (unsigned i = 0; i < probationaryTrackIndices.size(); i++) {
        Track &track = m_trackList[probationaryTrackIndices[i]];
        if (track.getState() == Track::STATE_DISABLED) {
            m_trackList.free(probationaryTrackIndices[i]);
            continue;
        }

        TrackObservation *nonRefObs = NULL;
        for (unsigned j = 0; j < track.getObservations().reservedSize(); j++) {
            if (!track.getObservations().inUse(j)) continue;

            if (!track.getObservations()[j].isReferenceObs()) {
                nonRefObs = &track.getObservations()[j];
                break;
            }
        }
        assert(nonRefObs != NULL);

        if (nonRefObs->getReferenceAlignmentError() > 1.5f) {
            track.switchState(Track::STATE_DISABLED);
            m_trackList.free(probationaryTrackIndices[i]);
        } else {
            numTracksAccepted++;
            track.updatePreWarpedDescriptor(NULL);
        }
    }
#else
    unsigned numTracksAccepted = probationaryTrackIndices.size();
    for (unsigned i = 0; i < probationaryTrackIndices.size(); i++) {
        Track &track = m_trackList[probationaryTrackIndices[i]];
        track.updateTrackMatchingDescriptor(NULL);
    }
#endif
    m_performanceStatistics.stopTimer(PerformanceStatistics::TIMER_NEW_TRACK_TESTING);

    if (m_config.verbatimLevel >= 2)
        std::cout << "New tracks: " << numTracksAccepted << std::endl;
    mergeDuplicateTracks();
}



void SFM::removeOutliers()
{
    AddCudaScopedProfileInterval("SFM::removeOutliers()");

    float maxError = 0.0f;
    float avgSum = 0.0f;
    unsigned avgSumDivisor = 0;
    std::vector<float> errors;
    errors.reserve(m_trackList.reservedSize());
    unsigned errorIndex;

    errorIndex = 0;
    for (Track &track : m_trackList) {
        if (track.getState() == Track::STATE_DISABLED)
            continue;
        for (TrackObservation &observation : track.getObservations()) {
            if (observation.isFaulty())
                continue;

            errors.push_back(observation.computeSQRScreenSpaceError() *
                             observation.getOverallWeight() * observation.getOverallWeight());
            maxError = std::max(maxError, errors[errorIndex]);

            if (track.getState() == Track::STATE_ACTIVE) {
                avgSum += errors[errorIndex];
                avgSumDivisor++;
            }

            errorIndex++;
        }
    }

    const float avg = avgSum / avgSumDivisor;
    //std::cout << "avg patch pixel error: " << sqrtf(avg) << std::endl;
    //const float thresh = fmax(avg * 4.5f, 1.0f);
    const float thresh1 = fmax(avg * m_config.outlierRemovalActiveThreshFactor, m_config.outlierRemovalActiveThreshMinimum);
    const float thresh2 = fmax(avg * m_config.outlierRemovalSemiActiveThreshFactor, m_config.outlierRemovalSemiActiveThreshMinimum);


    errorIndex = 0;
    for (Track &track : m_trackList) {
        if (track.getState() == Track::STATE_DISABLED)
            continue;

        const float thresh = (track.getState() == Track::STATE_ACTIVE)?thresh1:thresh2;

        bool changed = false;

        for (TrackObservation &observation : track.getObservations()) {
            if (observation.isFaulty())
                continue;

            if (errors[errorIndex] > thresh) {
                observation.markFaulty();
                changed = true;
            }
            errorIndex++;
        }
        if (changed)
            track.updateTrackMatchingDescriptor(nullptr);

    }
}

#ifdef SFM_PERFORM_PATCH_ALIGNMENT
void SFM::computeMissingFeaturePointAlignments(bool measureTime)
{
    AddCudaScopedProfileInterval("SFM::computeMissingFeaturePointAlignments");
    if (measureTime)
        m_performanceStatistics.startTimer(PerformanceStatistics::TIMER_TRACK_ALIGNMENT);
#if 1
    for (unsigned i = 0; i < m_trackList.reservedSize(); i++) {
        if (m_trackList.inUse(i)) {
            Track &track = m_trackList[i];
            if (track.getState() == Track::STATE_DISABLED)
                continue;

            track.chooseNewReferenceObservation();
        }
    }
#endif


    std::vector<FeaturePointAlignmentEstimator::FeaturePointPair> featurePointAlignmentPairs;
    featurePointAlignmentPairs.reserve(10000);
    std::vector<TrackObservation*> correspondingObservations;
    correspondingObservations.reserve(10000);

    for (unsigned i = 0; i < m_trackList.reservedSize(); i++) {
        if (!m_trackList.inUse(i))
            continue;

        Track &track = m_trackList[i];
        if (track.getState() == Track::STATE_DISABLED)
            continue;

        for (unsigned j = 0; j < track.getObservations().reservedSize(); j++) {
            if (!track.getObservations().inUse(j))
                continue;

            TrackObservation &obs = track.getObservations()[j];
            if (obs.isFaulty())
                continue;

            assert(track.getReferenceObservation() != NULL);
            if (&obs == track.getReferenceObservation())
                continue;

            if (obs.referenceObsHomographyValid())
                continue;

            TrackObservation &templateObs = *track.getReferenceObservation();


            FeaturePointAlignmentEstimator::FeaturePointPair FPPair;

            FPPair.templateFPFrame = templateObs.getFrame();
            FPPair.templateFPIndex = templateObs.getFrameFeaturePointIndex();
            FPPair.templateFPScreenSpacePosition = templateObs.getScreenSpacePosition();


            FPPair.secondFPFrame = obs.getFrame();
            FPPair.secondFPIndex = obs.getFrameFeaturePointIndex();
            FPPair.secondFPScreenSpacePosition = obs.getScreenSpacePosition();

            FPPair.normal = track.getNormal();
            FPPair.estimatedTrackPosition = track.getLastWSPositionEstimate();

            featurePointAlignmentPairs.push_back(FPPair);
            correspondingObservations.push_back(&obs);
        }
    }


    m_featurePointAlignmentEstimator.process(featurePointAlignmentPairs);

    unsigned failures = 0;
    for (unsigned i = 0; i < correspondingObservations.size(); i++) {
        const FeaturePointAlignmentEstimator::FeaturePointPair &FPPair = featurePointAlignmentPairs[i];

        float movement = std::sqrt((FPPair.secondFPScreenSpacePosition - correspondingObservations[i]->getScreenSpacePosition()).SQRLen());

        if ((FPPair.residualError > 1.5f) || (movement > 10/2000.0f)) {
            correspondingObservations[i]->markFaulty();
            correspondingObservations[i]->getTrack()->updatePreWarpedDescriptor(NULL);
            failures++;
        } else {
            correspondingObservations[i]->setScreenSpacePosition(FPPair.secondFPScreenSpacePosition);
            correspondingObservations[i]->setReferenceObsScreenToThisScreen(FPPair.resultingHomography, FPPair.residualError);
        }
    }

    std::cout << failures << " feature points failed realignment!" << std::endl;

    for (unsigned i = 0; i < m_trackList.reservedSize(); i++) {
        if (!m_trackList.inUse(i))
            continue;

        Track &track = m_trackList[i];
        if (track.getState() == Track::STATE_DISABLED)
            continue;

        for (unsigned j = 0; j < track.getObservations().reservedSize(); j++) {
            if (!track.getObservations().inUse(j))
                continue;

            TrackObservation &obs = track.getObservations()[j];
            if (obs.isFaulty())
                continue;

            assert(track.getReferenceObservation() != NULL);
            if (&obs == track.getReferenceObservation())
                continue;


            assert(obs.referenceObsHomographyValid());

        }
    }
    if (measureTime)
        m_performanceStatistics.stopTimer(PerformanceStatistics::TIMER_TRACK_ALIGNMENT);
}


void SFM::updateTrackNormals()
{
    AddCudaScopedProfileInterval("update track normals");

    for (unsigned i = 0; i < m_trackList.reservedSize(); i++) {
        if (m_trackList.inUse(i)) {
            Track &track = m_trackList[i];

            if (track.getState() == Track::STATE_DISABLED)
                continue;

            if (!track.needsNormalUpdate())
                continue;

            track.updateNormal();
        }
    }
}
#endif

void SFM::recomputeTrackWeights()
{
    AddCudaScopedProfileInterval("SFM::recomputeTrackWeights()");

    for (Track &track : m_trackList) {
        if (track.getState() == Track::STATE_DISABLED)
            continue;
#if 0

        unsigned numObs = 0;
        for (unsigned j = 0; j < track.getObservations().reservedSize(); j++) {
            if (track.getObservations().inUse(j)) {
                TrackObservation &observation = track.getObservations()[j];
                if (!observation.isFaulty()) {
                    numObs++;
                }
            }
        }

        track.setTrackWeight(numObs);
#else
        track.setTrackWeight(1.0f);
#endif
    }
}

void SFM::readjustSemiActiveTrackRange(unsigned start, unsigned count)
{
    std::vector<SingleTrackBundleAdjustment::Observation> observations;

    SingleTrackBundleAdjustment singleTrackBA;

    for (unsigned i = start; i < start+count; i++)
        if (m_trackList.inUse(i))
            if (m_trackList[i].getState() == Track::STATE_SEMI_ACTIVE) {
                Track &track = m_trackList[i];

                observations.clear();

                for (unsigned j = 0; j < track.getObservations().reservedSize(); j++) {
                    if (!track.getObservations().inUse(j))
                        continue;
                    TrackObservation &observation = track.getObservations()[j];

                    if (observation.isFaulty())
                        continue;


                    SingleTrackBundleAdjustment::Observation obs;
                    obs.projectionViewMatrix = &observation.getFrame()->getCamera().getProjectionViewMatrix3x4();
                    obs.distortion = &observation.getFrame()->getCamera().getInternalCalibration()->getRadialDistortion();
                    obs.screenSpacePosition = observation.getScreenSpacePosition();
                    obs.weight = observation.getOverallWeight();

                    observations.push_back(obs);
                }

                LinAlg::Vector4f trackPos = track.getLastWSPositionEstimate();
                singleTrackBA.optimize(observations, trackPos);
                track.setWSPosition(trackPos);
            }
}


void SFM::readjustSemiActiveTracks()
{
#if 0
    readjustSemiActiveTrackRange(0, m_trackList.reservedSize());
#else
    TaskGroup group;
    for (unsigned i = 0; i < m_trackList.reservedSize(); i+= m_config.baSemiActiveTracksBatchSize) {
        group.add(boost::bind(&SFM::readjustSemiActiveTrackRange, this, i, std::min(m_config.baSemiActiveTracksBatchSize, m_trackList.reservedSize() - i)),
                  TaskScheduler::get());
    }
    TaskScheduler::get().waitFor(&group);
#endif
}


void  SFM::checkMajorTrackChanges()
{
    for (Track &track : m_trackList)
        if (track.getState() != Track::STATE_DISABLED)
            track.checkForMajorChange(m_timestampClock);

}

void SFM::operate()
{
    if (m_state == OP_STATE_DONE)
        return;
    m_timestampClock++;
    iterateBA(200);
    if (m_bundleAdjustment.converged()) {
        switch (m_state) {
            case OP_STATE_ADDPATCHES:
#ifdef SFM_PERFORM_PATCH_ALIGNMENT
                computeMissingFeaturePointAlignments();
                updateTrackNormals();
#endif
                findAdditionalObservations();
#ifdef SFM_PERFORM_PATCH_ALIGNMENT
                computeMissingFeaturePointAlignments();
                updateTrackNormals();
#endif
                recomputeTrackWeights();
                m_state = OP_STATE_REMOVEOUTLIERS;
                m_removeOutlierRun = 0;
            break;
            case OP_STATE_REMOVEOUTLIERS:
                removeOutliers();
                if (++m_removeOutlierRun > m_config.numRemoveOutlierItertions) {
                    recomputeTrackWeights();
                    selectBASubset();
                    m_state = OP_STATE_ADDVIEW;

                    bool allViewsReconstructed = true;
                    for (unsigned i = 0; i < m_frames.size(); i++) {
                        if (!m_frames[i]->active()) {
                            allViewsReconstructed = false;
                            break;
                        }
                    }

                    if (!allViewsReconstructed && (m_numCyclesWithoutNewImages > m_config.maxRoundsWithoutNewImage)) {
                        m_state = OP_STATE_DONE;
                        m_performanceStatistics.startNewFrame();
                    }

                    if (allViewsReconstructed && (m_allViewsReconstructedExtraRounds++ == m_config.numExtraRounds)) {
                        m_state = OP_STATE_DONE;
                        m_performanceStatistics.startNewFrame();
#if 0
                        dumpPerfStats();

                        std::cout << "Matching stats: " << std::endl;
                        std::cout << "   Initial:" << std::endl;
                        std::cout << "       Num: " << m_descriptorMatchingStats.getNumMatchingOperations(DescriptorMatchingStats::MATCH_INITIAL) << std::endl;
                        std::cout << "       Time: " << m_descriptorMatchingStats.getNanosecondsSpendMatching(DescriptorMatchingStats::MATCH_INITIAL) * 1e-9 << std::endl;
                        std::cout << "       Avg: " << m_descriptorMatchingStats.getNanosecondsSpendMatching(DescriptorMatchingStats::MATCH_INITIAL) * 1e-9 /
                                                            m_descriptorMatchingStats.getNumMatchingOperations(DescriptorMatchingStats::MATCH_INITIAL) << std::endl;

                        std::cout << "   Unconstrained:" << std::endl;
                        std::cout << "       Num: " << m_descriptorMatchingStats.getNumMatchingOperations(DescriptorMatchingStats::MATCH_UNCONSTRAINED) << std::endl;
                        std::cout << "       Time: " << m_descriptorMatchingStats.getNanosecondsSpendMatching(DescriptorMatchingStats::MATCH_UNCONSTRAINED) * 1e-9 << std::endl;
                        std::cout << "       Avg: " << m_descriptorMatchingStats.getNanosecondsSpendMatching(DescriptorMatchingStats::MATCH_UNCONSTRAINED) * 1e-9 /
                                                            m_descriptorMatchingStats.getNumMatchingOperations(DescriptorMatchingStats::MATCH_UNCONSTRAINED) << std::endl;

                        std::cout << "   Epipolar:" << std::endl;
                        std::cout << "       Num: " << m_descriptorMatchingStats.getNumMatchingOperations(DescriptorMatchingStats::MATCH_EPIPOLAR) << std::endl;
                        std::cout << "       Time: " << m_descriptorMatchingStats.getNanosecondsSpendMatching(DescriptorMatchingStats::MATCH_EPIPOLAR) * 1e-9 << std::endl;
                        std::cout << "       Avg: " << m_descriptorMatchingStats.getNanosecondsSpendMatching(DescriptorMatchingStats::MATCH_EPIPOLAR) * 1e-9 /
                                                            m_descriptorMatchingStats.getNumMatchingOperations(DescriptorMatchingStats::MATCH_EPIPOLAR) << std::endl;

                        std::cout << "   Location:" << std::endl;
                        std::cout << "       Num: " << m_descriptorMatchingStats.getNumMatchingOperations(DescriptorMatchingStats::MATCH_LOCATION) << std::endl;
                        std::cout << "       Time: " << m_descriptorMatchingStats.getNanosecondsSpendMatching(DescriptorMatchingStats::MATCH_LOCATION) * 1e-9 << std::endl;
                        std::cout << "       Avg: " << m_descriptorMatchingStats.getNanosecondsSpendMatching(DescriptorMatchingStats::MATCH_LOCATION) * 1e-9 /
                                                            m_descriptorMatchingStats.getNumMatchingOperations(DescriptorMatchingStats::MATCH_LOCATION) << std::endl;

                        std::cout << "Full graph:" << std::endl;
                        analyzeMinCut(false);
                        std::cout << "Active graph:" << std::endl;
                        analyzeMinCut(true);

                        std::cout << "P = " << getInternalCameraCalibrations()[0]->getProjectionMatrix() << std::endl;

#if 1
saveCameraPosesToXml("cameraPoses.xml");
saveSparseReconstructionToBinFormat("sparseReconstruction.binSFM");
savePMVSFiles("pmvs");
exit(0);
#endif
#endif
                    }
                }
            break;
            case OP_STATE_ADDVIEW:
#ifdef SFM_PERFORM_PATCH_ALIGNMENT
                computeMissingFeaturePointAlignments();
                updateTrackNormals();
#endif
                activateNextFrames();
                recomputeTrackWeights();
                selectBASubset();
                m_state = OP_STATE_ADDPATCHES;
            break;
            default:
                throw std::runtime_error("Invalid op state!");
        }
        m_bundleAdjustment.restart();
    }

    m_performanceStatistics.startNewFrame();
}

void SFM::readbackBAData()
{
    for (auto &calib : m_internalCameraCalibrations) {
        if (calib->activeInBA()) {
            calib->readBackFromBA();
        }
    }

    for (auto &frame : m_frames) {
        if (frame->getCamera().activeInBA()) {
            frame->getCamera().readBackFromBA(m_bundleAdjustment);
        }
    }

    for (Track &track : m_trackList)
        if (track.getState() == Track::STATE_ACTIVE)
            track.readBackFromBA();
}

float SFM::computeSumSqrProjectionErrors()
{
    double sum = 0.0;

    for (Track &track : m_trackList)
        if (track.getState() == Track::STATE_ACTIVE)
            for (TrackObservation &observation : track.getObservations())
                if (!observation.isFaulty())
                    sum += observation.computeSQRScreenSpaceError();

    return sum;
}

float SFM::computeAvgSqrProjectionErrors()
{
    double sum = 0.0;
    unsigned count = 0;

    for (Track &track : m_trackList)
        if (track.getState() == Track::STATE_ACTIVE)
            for (TrackObservation &observation : track.getObservations())
                if (!observation.isFaulty()) {
                    float pureError = observation.computeSQRScreenSpaceError();
                    sum += pureError;
                    count++;
                }

    return sum / count;
}


void SFM::selectBASubset()
{
    AddCudaScopedProfileInterval("SFM::selectBASubset()");

    for (Track &track : m_trackList)
            track.clearBASubsetVotes();

    for (unsigned i = 0; i < m_frames.size(); i++) {
        if (m_frames[i]->active()) {
            m_frames[i]->voteForBASubset(m_config.subsetSelectionConfig);
        }
    }

    for (Track &track : m_trackList) {
        if (track.getState() != Track::STATE_DISABLED) {
            if (track.getBASubsetVotes() == 0)
                track.switchState(Track::STATE_SEMI_ACTIVE);
            else
                track.switchState(Track::STATE_ACTIVE);

        }
    }

}

void SFM::dumpTrackObsForAnalysis(const std::string &filename)
{
    std::fstream file(filename, std::ios_base::out);

    unsigned index1 = -1;
    unsigned index2 = -1;
    for (unsigned i = 0; i < m_frames.size(); i++) {
        if (m_frames[i]->active()) {
            index1 = i;
            break;
        }
    }
    if (index1 == (unsigned)-1)
        throw std::runtime_error("No frame active!");

    for (unsigned i = index1+1; i < m_frames.size(); i++) {
        if (m_frames[i]->active()) {
            index2 = i;
            break;
        }
    }
    if (index2 == (unsigned)-1)
        throw std::runtime_error("No second frame active!");

    file << index1 << " " << index2 << std::endl;
    file << "0" << std::endl;
    file << "1" << std::endl;
    file << "0.0" << std::endl;

    std::vector<LinAlg::Vector4f> pairs;

    for (unsigned i = 0; i < m_trackList.reservedSize(); i++) {
        if (!m_trackList.inUse(i))
            continue;

        const Track &track = m_trackList[i];
        if (track.getState() != Track::STATE_DISABLED) {
            boost::optional<LinAlg::Vector2f> screen1;
            boost::optional<LinAlg::Vector2f> screen2;
            for (unsigned j = 0; j < track.getObservations().reservedSize(); j++) {
                if (!track.getObservations().inUse(j))
                    continue;

                const TrackObservation &obs = track.getObservations()[j];
                if (obs.isFaulty())
                    continue;
/*
                if (obs.getWeight() < (1.0f / (1.0f / 1500.0f)))
                    continue;
*/
                if (obs.getFrame() == m_frames[index1].get()) {
//                    if (obs.getFrame()->getFeaturePoint(obs.getFrameFeaturePointIndex()).size < 3.0f / 4000.0f)
                        screen1 = obs.getScreenSpacePosition();
                }
                if (obs.getFrame() == m_frames[index2].get()) {
//                    if (obs.getFrame()->getFeaturePoint(obs.getFrameFeaturePointIndex()).size < 3.0f / 4000.0f)
                        screen2 = obs.getScreenSpacePosition();
                }
            }
            if (screen1 && screen2) {
                pairs.push_back(LinAlg::Fill((*screen1)[0] * 3072.0f*0.5f + 3072.0f*0.5f,
                                             (*screen1)[1] * 3072.0f*0.5f + 2048.0f*0.5f,
                                             (*screen2)[0] * 3072.0f*0.5f + 3072.0f*0.5f,
                                             (*screen2)[1] * 3072.0f*0.5f + 2048.0f*0.5f));
            }
        }
    }

    file << pairs.size() << std::endl;
    for (unsigned i = 0; i < pairs.size(); i++)
        file << pairs[i][0] << " " << pairs[i][1] << " " << pairs[i][2] << " " << pairs[i][3] << std::endl;

}





}
