/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef CUDASIFT_H
#define CUDASIFT_H

#include "../cudaUtilities/CudaCodeModule.h"
#include "../cudaUtilities/CudaKernel.h"

#include "../cudaUtilities/CudaTextureMemory.h"
#include "../cudaUtilities/CudaDeviceMemory.h"

#include "../cudaUtilities/CudaTextureReference.h"
#include "../cudaUtilities/CudaSurfaceReference.h"

#include <memory>

#include "../tools/RasterImage.h"

#include "../cudaKernels/CudaSift.cuh"

#include "PatchCacheTypes.h"


namespace SFM {
namespace config {

struct FeatureExtractionConfig;
struct CudaConfig;

}
}


class CudaSift
{
    public:
        CudaSift(const SFM::config::FeatureExtractionConfig &config, const SFM::config::CudaConfig &cudaConfig);

        struct FeaturePoint {
            float angle;
            float scale;
            float x;
            float y;
            unsigned layer;

            unsigned char descriptor[128] __attribute__((aligned (16)));
            unsigned sign;

            float locationPrecision;

#ifdef CudaSift_EXTRACT_PATCH_DATA
            PackedPatch packedPatch;
#endif
        };

        void gatherFeaturePoints(const RasterImage &image, std::vector<FeaturePoint> &featurePoints);
    protected:
        const SFM::config::FeatureExtractionConfig &m_config;
        CudaUtils::CudaCodeModule m_codeModule;

        std::unique_ptr<CudaUtils::CudaKernel> m_debugExtractHalfFloatFromLayeredArrayKernel;
        std::unique_ptr<CudaUtils::CudaKernel> m_extractLuminanceKernel;
        std::unique_ptr<CudaUtils::CudaKernel> m_downsampleKernel;
        std::unique_ptr<CudaUtils::CudaKernel> m_locatePossibleFeaturePointsKernel;
        std::unique_ptr<CudaUtils::CudaKernel> m_extractFeaturePointsKernel;

        enum GaussianFilterKernelTypes {
            guassianFilterH_2,
            guassianFilterV_2,
            guassianFilterH_3,
            guassianFilterV_3,
            guassianFilterH_4,
            guassianFilterV_4,
            guassianFilterH_5,
            guassianFilterV_5,
            guassianFilterH_6,
            guassianFilterV_6,
            guassianFilterH_7,
            guassianFilterV_7,
            guassianFilterH_8,
            guassianFilterV_8,
            GuassianFilterKernelCount
        };
        std::unique_ptr<CudaUtils::CudaKernel> m_gaussFilterKernel[GuassianFilterKernelCount];
        inline CudaUtils::CudaKernel* getGaussianFilterKernel(bool horizontal, unsigned size) {
            if (size < 2)
                return NULL;
            if (size > 8)
                return NULL;
            return m_gaussFilterKernel[(size-2)*2+(horizontal?0:1)].get();
        }
        /*
        std::unique_ptr<CudaUtils::CudaKernel> m_absMaxHKernel;
        std::unique_ptr<CudaUtils::CudaKernel> m_absMaxVKernel;
        std::unique_ptr<CudaUtils::CudaKernel> m_locatePossibleFeaturePointsKernel;
        std::unique_ptr<CudaUtils::CudaKernel> m_extractFeaturePointsKernel;
        */
        CudaUtils::CudaTextureMemory m_sourceRGBAImage;

        std::unique_ptr<CudaUtils::CudaTextureReference> m_sourceRGBAImageTexRef;
        std::unique_ptr<CudaUtils::CudaTextureReference> m_sourceLumImageTexRef;
        std::unique_ptr<CudaUtils::CudaTextureReference> m_sourcePrevGaussImageTexRef;
        std::unique_ptr<CudaUtils::CudaTextureReference> m_sourceTmpImageTexRef;
        std::unique_ptr<CudaUtils::CudaTextureReference> m_debugLayeredArrayTexRef;
        std::unique_ptr<CudaUtils::CudaTextureReference> m_sourceDoGImageTexRef;

        std::unique_ptr<CudaUtils::CudaSurfaceReference> m_outputTmpSurfRef;
        std::unique_ptr<CudaUtils::CudaSurfaceReference> m_outputGaussianImageSurfRef;
        std::unique_ptr<CudaUtils::CudaSurfaceReference> m_outputDifferenceOfGaussianImageSurfRef;


        std::unique_ptr<CudaUtils::CudaConstantMemory> m_filterKernelScales;
        std::unique_ptr<CudaUtils::CudaConstantMemory> m_filterKernelOffsets;

        std::unique_ptr<CudaUtils::CudaSurfaceReference> m_downsampleInputSurfRef;
        std::unique_ptr<CudaUtils::CudaSurfaceReference> m_downsampleOutputSurfRef;

        CudaUtils::CudaDeviceMemory m_possibleFeaturepointLocationArray;
        CudaUtils::CudaDeviceMemory m_featurePointArray;

/*
        CudaUtils::CudaDeviceMemory m_featurepointArray;

        struct Level {
            float scale;
            float rcpScale;
            float idealWidth;
            float idealHeight;
            float idealRcpWidth;
            float idealRcpHeight;
            unsigned texture;
            unsigned lod;
        };

        std::vector<Level> m_level;

        void resize(unsigned width, unsigned height);

        void dumpPyramid(const std::string &baseFilename, CudaUtils::CudaMipmappedTexture *pyramid, float scale = 1.0f, std::vector<FeaturePoint> *overlay = NULL);
        void dumpMinMaxPyramid(const std::string &baseFilename, CudaUtils::CudaMipmappedTexture *pyramid, float scale = 1.0f);
*/

        void dumpLayer(const std::string &filename, CudaUtils::CudaMipmappedTexture *texture, unsigned layer, float scale = 1.0f, unsigned numFP = 0, CudaSiftPossibleFeaturePointLocation *locations = NULL);
        void dumpTexture(const std::string &filename, CudaUtils::CudaMipmappedTexture *texture, float scale = 1.0f);

};

#endif // CUDASIFT_H
