/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "HomographyImageResampler.h"

#include "../cudaKernels/HomographyImageResampling.cuh"

HomographyImageResampler::HomographyImageResampler()
{
    m_codeModule.loadFromFile("../SFMBackend/kernels/Release/HomographyImageResampling.fatbin");

    m_kernel = std::unique_ptr<CudaUtils::CudaKernel>(m_codeModule.getKernel("HomographyImageResampling"));


    m_sourceImageTexRef = std::unique_ptr<CudaUtils::CudaTextureReference>(m_codeModule.getTexReference("sourceImage"));
    m_sourceImageTexRef->setTexelFilterMode(CudaUtils::CudaTextureReference::FILTER_MODE_LINEAR);
    m_sourceImageTexRef->setMipmapFilterMode(CudaUtils::CudaTextureReference::FILTER_MODE_LINEAR);
    m_sourceImageTexRef->setCoordinateNormalization(true);

}

void HomographyImageResampler::resample(CudaUtils::CudaMipmappedTexture &src, RasterImage &dst, const LinAlg::Matrix3x3f &homography, float lodBias)
{

    m_sourceImageTexRef->bindMipmappedTexture(&src);
    m_sourceImageTexRef->setMinMaxMipLevel(0, src.getNumLevel()-1);

    m_outputArray.resize(dst.getWidth() * dst.getHeight() * 4);


    HomographyImageResamplingKernelParams kernelParams;
    kernelParams.srcWidth = src.getLevel(0).getWidth();
    kernelParams.srcHeight = src.getLevel(0).getHeight();
    kernelParams.width = dst.getWidth();
    kernelParams.height = dst.getHeight();
    kernelParams.LODbias = lodBias;
    kernelParams.dstRGBA = (unsigned*) m_outputArray.getPtr();

    LinAlg::Matrix3x3f hom = homography * LinAlg::Scale2D(LinAlg::Fill(1.0f / dst.getWidth(), 1.0f / dst.getHeight()));
    for (unsigned i = 0; i < 3; i++)
        for (unsigned j = 0; j < 3; j++)
            kernelParams.homography[i*3+j] = hom[i][j];

    m_kernel->launch(LinAlg::Fill(16u, 16u, 1u),
                        LinAlg::Fill((kernelParams.width+15u)/16u, (kernelParams.height+15u)/16u, 1u),
                        &kernelParams, sizeof(kernelParams));

    m_outputArray.download(dst.getData(), dst.getWidth() * dst.getHeight() * 4);
}
