/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "ImageMipchainBuilder.h"
#include "../cudaUtilities/cudaProfilingScope.h"

ImageMipchainBuilder::ImageMipchainBuilder()
{
    m_codeModule.loadFromFile("../SFMBackend/kernels/Release/buildMipmapChain.fatbin");

    m_downsampleRGBMImageKernel = std::unique_ptr<CudaUtils::CudaKernel>(m_codeModule.getKernel("downsampleRGBMImage"));
    m_convertRGBMImageKernel = std::unique_ptr<CudaUtils::CudaKernel>(m_codeModule.getKernel("convertRGBMImage"));
    m_mipChainInputTexRef = std::unique_ptr<CudaUtils::CudaTextureReference>(m_codeModule.getTexReference("sourceImage"));
    m_mipChainOutputSurfRef = std::unique_ptr<CudaUtils::CudaSurfaceReference>(m_codeModule.getSurfReference("outputImage"));

    m_mipChainInputTexRef->setTexelFilterMode(CudaUtils::CudaTextureReference::FILTER_MODE_LINEAR);
    m_mipChainInputTexRef->setCoordinateNormalization(true);
}

ImageMipchainBuilder::~ImageMipchainBuilder()
{
    //dtor
}

struct DownsampleRGBMImageKernelParams {
    unsigned width;
    unsigned height;
    float scaleX;
    float scaleY;
};

void ImageMipchainBuilder::buildMipChain(const RasterImage &source, CudaUtils::CudaMipmappedTexture &destination)
{
    AddCudaScopedProfileInterval("ImageMipchainBuilder::buildMipChain");

    destination.resize(source.getWidth(), source.getHeight(), 0, CU_AD_FORMAT_UNSIGNED_INT8, 4, CUDA_ARRAY3D_SURFACE_LDST);

    destination.getLevel(0).syncUploadAll(source.getData(), 4*source.getWidth());

    {
        unsigned conversionParams[2];
        conversionParams[0] = destination.getLevel(0).getWidth();
        conversionParams[1] = destination.getLevel(0).getHeight();

        m_mipChainOutputSurfRef->bindTexture(&destination.getLevel(0));

        m_convertRGBMImageKernel->launch(LinAlg::Fill(16u, 16u, 1u),
                                          LinAlg::Fill((conversionParams[0]+15u)/16u, (conversionParams[1]+15u)/16u, 1u),
                                          &conversionParams, sizeof(unsigned)*2);

    }

    for (unsigned i = 1; i < destination.getNumLevel(); i++) {
        m_mipChainInputTexRef->bindTexture(&destination.getLevel(i-1));
        m_mipChainOutputSurfRef->bindTexture(&destination.getLevel(i));

        DownsampleRGBMImageKernelParams params;
        params.width = destination.getLevel(i).getWidth();
        params.height = destination.getLevel(i).getHeight();
/*
        params.scaleX = destination.getLevel(i-1).getWidth() / (float) destination.getLevel(i).getWidth();
        params.scaleY = destination.getLevel(i-1).getHeight() / (float) destination.getLevel(i).getHeight();
*/
        params.scaleX = 1.0f / (float) destination.getLevel(i).getWidth();
        params.scaleY = 1.0f / (float) destination.getLevel(i).getHeight();

        m_downsampleRGBMImageKernel->launch(LinAlg::Fill(16u, 16u, 1u),
                                          LinAlg::Fill((params.width+15u)/16u, (params.height+15u)/16u, 1u),
                                          &params, sizeof(params));
    }

#if 0
    for (unsigned i = 0; i < destination.getNumLevel(); i++) {
        RasterImage dstImage;
        dstImage.resize(destination.getLevel(i).getWidth(), destination.getLevel(i).getHeight());
        destination.getLevel(i).syncDownloadAll(dstImage.getData(), 4*dstImage.getWidth());

        for (unsigned i = 0; i < dstImage.getWidth() * dstImage.getHeight(); i++) {

            unsigned char *pixel = (unsigned char*)&dstImage.getData()[i];
            pixel[0] = pixel[0] * pixel[3]/255;
            pixel[1] = pixel[1] * pixel[3]/255;
            pixel[2] = pixel[2] * pixel[3]/255;
            pixel[3] = 255;
        }

        std::stringstream str;
        str << "debugOutput"<<i<<".png";

        dstImage.writeToFile(str.str().c_str());
    }
#endif

}
