/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "PackedPatchToPatchCacheTransfer.h"
#include <assert.h>
#include "../cudaKernels/PackedPatchToPatchCache.cuh"

PackedPatchToPatchCacheTransfer::PackedPatchToPatchCacheTransfer()
{
    m_codeModule.loadFromFile("../SFMBackend/kernels/Release/PackedPatchToPatchCache.fatbin");

    m_kernel = std::unique_ptr<CudaUtils::CudaKernel>(m_codeModule.getKernel("PackedPatchToPatchCacheKernel"));

    m_outputSurfRef = std::unique_ptr<CudaUtils::CudaSurfaceReference>(m_codeModule.getSurfReference("PatchCache"));
}

PackedPatchToPatchCacheTransfer::~PackedPatchToPatchCacheTransfer()
{
    //dtor
}


void PackedPatchToPatchCacheTransfer::transfer(const std::vector<TransferData> &transferData, PatchCache &cache)
{
    m_patchData.resize(transferData.size() * PackedPatch::PACKED_PATCH_DATA_SIZE);
    m_transferData.resize(transferData.size() * sizeof(PackedPatchTransferMetaData));

    for (unsigned i = 0; i < transferData.size(); i++) {
        assert(transferData[i].slot.valid());
        unsigned slotIndex = transferData[i].slot.getSlotIndex();

        unsigned x, y, layer;
        x = cache.getCol(slotIndex);
        y = cache.getRow(slotIndex);
        layer = cache.getLayer(slotIndex);

        LinAlg::Vector2f scale =
                    LinAlg::Fill(1.0f / (transferData[i].packedPatch->screenTexelSizeX * PATCH_CACHE_LAYER_SIZE),
                                 1.0f / (transferData[i].packedPatch->screenTexelSizeY * PATCH_CACHE_LAYER_SIZE));

        LinAlg::Vector2f offset =
                    LinAlg::Fill(-transferData[i].packedPatch->screenCenterX * scale[0] + (x * PATCH_CACHE_PATCH_SIZE + PATCH_CACHE_PATCH_SIZE/2) / (float)PATCH_CACHE_LAYER_SIZE,
                                 -transferData[i].packedPatch->screenCenterY * scale[1] + (y * PATCH_CACHE_PATCH_SIZE + PATCH_CACHE_PATCH_SIZE/2) / (float)PATCH_CACHE_LAYER_SIZE);

        cache.getScreenSpaceToCacheSpaceScale(slotIndex) = scale;
        cache.getScreenSpaceToCacheSpaceOffset(slotIndex) = offset;

        PackedPatchTransferMetaData metaData;
        metaData.dstOffsetX = x * PATCH_CACHE_PATCH_SIZE;
        metaData.dstOffsetY = y * PATCH_CACHE_PATCH_SIZE;
        metaData.dstLayer = layer;

        m_patchData.upload(transferData[i].packedPatch->data, PackedPatch::PACKED_PATCH_DATA_SIZE, i*PackedPatch::PACKED_PATCH_DATA_SIZE);
        m_transferData.upload(&metaData, sizeof(PackedPatchTransferMetaData), i*sizeof(PackedPatchTransferMetaData));
    }

    {
        m_outputSurfRef->bindTexture(&cache.getAtlasTexture()->getLevel(0));

        PackedPatchToPatchCacheKernelParams kernelParams;
        kernelParams.metaData = (PackedPatchTransferMetaData*) m_transferData.getPtr();
        kernelParams.patchData = (unsigned char*) m_patchData.getPtr();

        m_kernel->launch(LinAlg::Fill(32u, 32u, 1u),
                         LinAlg::Fill((unsigned)transferData.size(), 1u, 1u),
                         &kernelParams, sizeof(PackedPatchToPatchCacheKernelParams));
    }
}
