/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "PatchCache.h"
#include <algorithm>

PatchCache::PatchCache()
{
    m_numLayers = 64;
    m_atlasTexture = std::unique_ptr<CudaUtils::CudaMipmappedTexture>(new CudaUtils::CudaMipmappedTexture());
    m_atlasTexture->resize(PATCH_CACHE_LAYER_SIZE, PATCH_CACHE_LAYER_SIZE, m_numLayers,
                           CU_AD_FORMAT_UNSIGNED_INT8, 1,
                           CUDA_ARRAY3D_SURFACE_LDST | CUDA_ARRAY3D_LAYERED, 1);

    unsigned numSlots = PATCH_CACHE_NUM_ROWS * PATCH_CACHE_NUM_ROWS * m_numLayers;
    m_slots.resize(numSlots);

    m_cachedLRUList.resize(numSlots);
    for (unsigned i = 0; i < numSlots; i++)
        m_cachedLRUList[i] = i;
    m_cachedLRUListStale = true;

    m_nextUserID = 0;
    m_nextTimestamp = 0;
}

PatchCache::~PatchCache()
{
    //dtor
}


void PatchCache::allocateSlots(SlotHandle *handles, unsigned count)
{
    if (m_cachedLRUListStale)
        rebuildLRUList();


    for (unsigned i = 0; i < count; i++) {
        if (m_slots[m_cachedLRUList[i]].lastUsed == (uint64_t)-1)
            throw std::runtime_error("Patch cache overflow!");

        m_slots[m_cachedLRUList[i]].userID = m_nextUserID++;
        handles[i] = SlotHandle(m_cachedLRUList[i], &m_slots[m_cachedLRUList[i]]);
    }
}

struct PatchCacheComparator {
    std::vector<PatchCache::Slot> &m_slots;
    PatchCacheComparator(std::vector<PatchCache::Slot> &slots) : m_slots(slots) { }
    bool operator()(unsigned lhs, unsigned rhs) const {
        return m_slots[lhs].lastUsed < m_slots[rhs].lastUsed;
    }
};

void PatchCache::rebuildLRUList()
{
    PatchCacheComparator patchCacheComparator(m_slots);
    std::sort(m_cachedLRUList.begin(), m_cachedLRUList.end(), patchCacheComparator);
    m_cachedLRUListStale = false;
}
