/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef PATCHCACHE_H
#define PATCHCACHE_H

#include <memory>
#include <string.h>

#include "../cudaUtilities/CudaTextureMemory.h"
#include "../cudaUtilities/CudaDeviceMemory.h"
#include "PatchCacheTypes.h"

class PatchCache
{
    public:
        PatchCache();
        ~PatchCache();

        struct Slot {
            uint64_t lastUsed;
            uint64_t userID;

            LinAlg::Vector2f screenSpaceToCacheSpaceScale;
            LinAlg::Vector2f screenSpaceToCacheSpaceOffset;

            Slot() : lastUsed(0), userID(0) { }
        };

        class SlotHandle {
            public:
                SlotHandle() : m_slotIndex(-1), m_slot(NULL), m_refID(-1) { }
                SlotHandle(unsigned slotIndex, Slot *slot) : m_slotIndex(slotIndex), m_slot(slot), m_refID(slot->userID) { }

                inline bool valid() const { return (m_slot != NULL) && (m_slot->userID == m_refID); }
                inline unsigned getSlotIndex() const { return m_slotIndex; }
            private:
                unsigned m_slotIndex;
                Slot *m_slot;
                uint64_t m_refID;
        };

        inline void lockSlot(unsigned slotIndex) { m_slots[slotIndex].lastUsed = -1; m_cachedLRUListStale = true; }
        inline void unlockSlot(unsigned slotIndex) { m_slots[slotIndex].lastUsed = m_nextTimestamp; m_cachedLRUListStale = true; }

        inline void slotUsed(unsigned slotIndex) { m_slots[slotIndex].lastUsed = m_nextTimestamp; m_cachedLRUListStale = true; }

        void allocateSlots(SlotHandle *handles, unsigned count);
        inline void incTimestamp() { m_nextTimestamp++; }

        LinAlg::Vector2f &getScreenSpaceToCacheSpaceScale(unsigned slotIndex) { return m_slots[slotIndex].screenSpaceToCacheSpaceScale; }
        const LinAlg::Vector2f &getScreenSpaceToCacheSpaceScale(unsigned slotIndex) const { return m_slots[slotIndex].screenSpaceToCacheSpaceScale; }

        LinAlg::Vector2f &getScreenSpaceToCacheSpaceOffset(unsigned slotIndex) { return m_slots[slotIndex].screenSpaceToCacheSpaceOffset; }
        const LinAlg::Vector2f &getScreenSpaceToCacheSpaceOffset(unsigned slotIndex) const { return m_slots[slotIndex].screenSpaceToCacheSpaceOffset; }

        inline unsigned getCol(unsigned slotIndex) const { return slotIndex % PATCH_CACHE_NUM_ROWS; }
        inline unsigned getRow(unsigned slotIndex) const { return (slotIndex / PATCH_CACHE_NUM_ROWS) % PATCH_CACHE_NUM_ROWS; }
        inline unsigned getLayer(unsigned slotIndex) const { return slotIndex / (PATCH_CACHE_NUM_ROWS*PATCH_CACHE_NUM_ROWS); }


        CudaUtils::CudaMipmappedTexture *getAtlasTexture() { return m_atlasTexture.get(); }
    protected:
        unsigned m_numLayers;
        std::vector<Slot> m_slots;
        std::unique_ptr<CudaUtils::CudaMipmappedTexture> m_atlasTexture;

        std::vector<unsigned> m_cachedLRUList;
        bool m_cachedLRUListStale;

        uint64_t m_nextTimestamp;
        uint64_t m_nextUserID;

        void rebuildLRUList();
};

#endif // PATCHCACHE_H
