/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "SourceImageToPatchAtlasTransfer.h"

#include "PatchAtlas.h"

#include "../cudaKernels/sourceImageToPatchAtlasTransfer.h"

SourceImageToPatchAtlasTransfer::SourceImageToPatchAtlasTransfer()
{
    m_codeModule.loadFromFile("../SFMBackend/kernels/Release/sourceImageToPatchAtlasTransfer.fatbin");

    m_kernel = std::unique_ptr<CudaUtils::CudaKernel>(m_codeModule.getKernel("transferLayerLevel"));
    m_sourceImageTexRef = std::unique_ptr<CudaUtils::CudaTextureReference>(m_codeModule.getTexReference("sourceImage"));

    m_sourceImageTexRef->setTexelFilterMode(CudaUtils::CudaTextureReference::FILTER_MODE_LINEAR);
    m_sourceImageTexRef->setMipmapFilterMode(CudaUtils::CudaTextureReference::FILTER_MODE_NEAREST);
    m_sourceImageTexRef->setCoordinateNormalization(true);

    m_kernelConstantParams = std::unique_ptr<CudaUtils::CudaConstantMemory>(m_codeModule.getConstantMemory("transferGlobals"));

    m_outputSurfRef = std::unique_ptr<CudaUtils::CudaSurfaceReference>(m_codeModule.getSurfReference("outputLayerLevel"));


    m_normalizationParameterKernel = std::unique_ptr<CudaUtils::CudaKernel>(m_codeModule.getKernel("computeNormalizationParams"));

}

SourceImageToPatchAtlasTransfer::~SourceImageToPatchAtlasTransfer()
{
    //dtor
}


void SourceImageToPatchAtlasTransfer::transfer(CudaUtils::CudaMipmappedTexture &sourceImage, std::vector<TransferData> &transferData, PatchAtlas *atlas)
{
    m_sourceImageTexRef->bindMipmappedTexture(&sourceImage);
    m_sourceImageTexRef->setMinMaxMipLevel(0.0f, sourceImage.getNumLevel()-1);


    std::vector<TransferInfo> cpuTransferInfo;
    TransferGlobals cpuTransferGlobals;

    cpuTransferInfo.resize(transferData.size());
    for (unsigned i = 0; i < transferData.size(); i++) {
        TransferInfo &info = cpuTransferInfo[i];
        info.sourceBaseMipLevel = (int) transferData[i].size;

        const float aspectRatio = (float)sourceImage.getLevel(0).getHeight() / (float)sourceImage.getLevel(0).getWidth();
        //const float rcpAspectRatio = 1.0f / (float)sourceImage.getLevel(0).getHeight() * (float)sourceImage.getLevel(0).getWidth();

#if 0
        const unsigned mipLevelWidth = sourceImage.getLevel(info.sourceBaseMipLevel).getWidth();
        const unsigned mipLevelHeight = sourceImage.getLevel(info.sourceBaseMipLevel).getHeight();


        const int pixelLocationX = (int)(transferData[i].x * mipLevelWidth * 0.5f + mipLevelWidth * 0.5f);
        const int pixelLocationY = (int)(transferData[i].y * rcpAspectRatio * mipLevelHeight * 0.5f + mipLevelHeight * 0.5f);

        info.sourceX = pixelLocationX / (float)mipLevelWidth;
        info.sourceY = pixelLocationY / (float)mipLevelHeight;
#else
        info.sourceX = transferData[i].x * 0.5f + 0.5f;
        info.sourceY = transferData[i].y * 0.5f / sourceImage.getLevel(0).getHeight() * sourceImage.getLevel(0).getWidth() + 0.5f;
#endif
        info.destinationX = atlas->getPatchParams()[transferData[i].atlasIndex].patchX * PatchAtlasConstants::ATLAS_PATCH_SIZE;
        info.destinationY = atlas->getPatchParams()[transferData[i].atlasIndex].patchY * PatchAtlasConstants::ATLAS_PATCH_SIZE;
        info.destinationLayer = atlas->getPatchParams()[transferData[i].atlasIndex].layer;
#if 0
        info.stepScaleX = sourceImage.getLevel(0).getWidth() / (float) mipLevelWidth;
        info.stepScaleY = sourceImage.getLevel(0).getHeight() / (float) mipLevelHeight;
#else
        info.stepScale = 1 << info.sourceBaseMipLevel;
#endif

        atlas->setSourceImageToAtlasPatchScaleOffset(transferData[i].atlasIndex,
                                                     info.sourceX * 2.0f - 1.0f,
                                                    (info.sourceY - 0.5f) * 2.0f * aspectRatio,
                                                     (PatchAtlasConstants::ATLAS_PATCH_SIZE << info.sourceBaseMipLevel) / (float) sourceImage.getLevel(0).getWidth(),
                                                     (PatchAtlasConstants::ATLAS_PATCH_SIZE << info.sourceBaseMipLevel) / (float) sourceImage.getLevel(0).getWidth());
    }

    m_videoData.resize(cpuTransferInfo.size() * sizeof(TransferInfo));
    m_videoData.upload(&cpuTransferInfo[0], cpuTransferInfo.size() * sizeof(TransferInfo));

    m_normalizationParameters.resize(2*4*cpuTransferInfo.size());

    {
        const unsigned level = 0;

        cpuTransferGlobals.w = PatchAtlasConstants::ATLAS_PATCH_SIZE >> level;
        cpuTransferGlobals.h = PatchAtlasConstants::ATLAS_PATCH_SIZE >> level;
        cpuTransferGlobals.wHalf = cpuTransferGlobals.w * 0.5f + 0.5f;
        cpuTransferGlobals.hHalf = cpuTransferGlobals.h * 0.5f + 0.5f;
        cpuTransferGlobals.mipLevel = level;
        cpuTransferGlobals.stepX = (1 << level) / (float)sourceImage.getLevel(0).getWidth();
        cpuTransferGlobals.stepY = (1 << level) / (float)sourceImage.getLevel(0).getHeight();

        m_kernelConstantParams->upload(&cpuTransferGlobals, sizeof(cpuTransferGlobals));

        m_outputSurfRef->bindTexture(&atlas->getAtlasTexture()->getLevel(level));

        void *ptr[2];
        ptr[0] = m_videoData.getPtr();
        ptr[1] = m_normalizationParameters.getPtr();
        m_normalizationParameterKernel->launch(LinAlg::Fill(8u, 8u, 1u),
                                      LinAlg::Fill((unsigned)cpuTransferInfo.size(), 1u, 1u),
                                      &ptr, sizeof(void*)*2);
    }


    for (unsigned level = 0; level < PatchAtlasConstants::ATLAS_MIP_LEVEL; level++) {
        cpuTransferGlobals.w = PatchAtlasConstants::ATLAS_PATCH_SIZE >> level;
        cpuTransferGlobals.h = PatchAtlasConstants::ATLAS_PATCH_SIZE >> level;
        cpuTransferGlobals.wHalf = cpuTransferGlobals.w * 0.5f + 0.5f;
        cpuTransferGlobals.hHalf = cpuTransferGlobals.h * 0.5f + 0.5f;
        cpuTransferGlobals.mipLevel = level;
        cpuTransferGlobals.stepX = (1 << level) / (float)sourceImage.getLevel(0).getWidth();
        cpuTransferGlobals.stepY = (1 << level) / (float)sourceImage.getLevel(0).getHeight();

        m_kernelConstantParams->upload(&cpuTransferGlobals, sizeof(cpuTransferGlobals));

        m_outputSurfRef->bindTexture(&atlas->getAtlasTexture()->getLevel(level));

        void *ptr[2];
        ptr[0] = m_videoData.getPtr();
        ptr[1] = m_normalizationParameters.getPtr();
        m_kernel->launch(LinAlg::Fill(8u, 8u, 1u),
                                      LinAlg::Fill((unsigned)cpuTransferInfo.size(), 1u, 1u),
                                      &ptr, sizeof(void*)*2);
    }
}
