/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
/**
 * @file
 * @author Andreas Ley
 */
#include "CudaDeviceMemory.h"
#include <cuda.h>
#include "CudaDriver.h"
#include <stdexcept>

namespace CudaUtils {


void BaseCudaDeviceMemory::upload(const void *src, ///< Pointer to the (possibly pinned) system memory location from where the data is to be taken from
                         size_t size,     ///< Amount of bytes to copy
                         size_t offset    ///< Offset into the video memory block from where to start writing
                         )
{
    CudaDriver::throwOnCudaError(cuMemcpyHtoD((CUdeviceptr)m_ptr + offset, src, size), __FILE__, __LINE__);
}

void BaseCudaDeviceMemory::download(void *dst,       ///< Pointer to the (possibly pinned) system memory location where the data is to be stored to
                           size_t size,     ///< Amount of bytes to copy
                           size_t offset    ///< Offset into the video memory block from where to start reading
                           ) const
{
    CudaDriver::throwOnCudaError(cuMemcpyDtoH(dst, (CUdeviceptr)m_ptr + offset, size), __FILE__, __LINE__);
}


void BaseCudaDeviceMemory::uploadAsync(const void *src, size_t size, size_t offset, const CudaStream &stream)
{
    CudaDriver::throwOnCudaError(cuMemcpyHtoDAsync((CUdeviceptr)m_ptr + offset, src, size, stream.getHandle()), __FILE__, __LINE__);
}

void BaseCudaDeviceMemory::downloadAsync(void *dst, size_t size, size_t offset, const CudaStream &stream) const
{
    CudaDriver::throwOnCudaError(cuMemcpyDtoHAsync(dst, (CUdeviceptr)m_ptr + offset, size, stream.getHandle()), __FILE__, __LINE__);
}



CudaConstantMemory::CudaConstantMemory(void *ptr, unsigned size)
{
    m_ptr = ptr;
    m_size = size;
    m_reserved = size;
}

CudaConstantMemory::~CudaConstantMemory()
{

}


void CudaConstantMemory::resize(size_t size)
{
    throw std::runtime_error("Can not resize constant memory!");
}




CudaDeviceMemory::CudaDeviceMemory()
{
}

CudaDeviceMemory::~CudaDeviceMemory()
{
	if (m_ptr != NULL)
		cuMemFree((CUdeviceptr)m_ptr);
}

void CudaDeviceMemory::resize(size_t size)
{
    if (size > m_reserved) {
        if (m_ptr != NULL)
            cuMemFree((CUdeviceptr)m_ptr);
        m_reserved = size;
        CudaDriver::throwOnCudaError(cuMemAlloc((CUdeviceptr*)&m_ptr, m_reserved), __FILE__, __LINE__);
    }
    m_size = size;
}


}
