/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
/**
 * @file
 * @author Andreas Ley
 */
#include "CudaFence.h"
#include <stdio.h>
#include <stdexcept>
#include <cuda.h>
#include <cuda_runtime_api.h>
#include <string>

typedef std::runtime_error CudaException;


CudaFence::CudaFence()
{
    cudaEventCreate(&m_event);
}

CudaFence::~CudaFence()
{
    cudaEventDestroy(m_event);
}


void CudaFence::insertIntoStream(const CudaStream &stream)
{
    if (cudaEventRecord(m_event, stream.getStream()) != cudaSuccess)
		throw CudaException((std::string("Inserting cuda fence failed: ") + cudaGetErrorString(cudaGetLastError())).c_str());
}

void CudaFence::insertIntoMainStream()
{
    if (cudaEventRecord(m_event, 0) != cudaSuccess)
		throw CudaException((std::string("Inserting cuda fence failed: ") + cudaGetErrorString(cudaGetLastError())).c_str());
}

bool CudaFence::query()
{
    switch (cudaEventQuery(m_event)) {
		case cudaSuccess:
			return true;
		case cudaErrorNotReady:
			return false;
		default:
			throw CudaException((std::string("Query on cuda fence failed: ") + cudaGetErrorString(cudaGetLastError())).c_str());
	}
}

void CudaFence::waitFor()
{
    if (cudaEventSynchronize(m_event) != cudaSuccess)
		throw CudaException((std::string("Waiting for cuda event failed: ") + cudaGetErrorString(cudaGetLastError())).c_str());
}
