/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef BLOCKSPARSEMATRIX_H
#define BLOCKSPARSEMATRIX_H

#include <vector>
#include <stdint.h>

class RasterImage;

namespace LinAlg {

/** @addtogroup Codebase_Group
 *  @{
 */

namespace BlockSparseMatrix {

enum {
    BLOCK_SIZE = 4
};

class LayoutGenerator
{
    public:
        void resize(unsigned w, unsigned h);
        void clear();
        inline void set(unsigned x, unsigned y) { m_blocksUsed[(y/BLOCK_SIZE)*m_numBlocksX + x/BLOCK_SIZE] = true; }
        inline bool isSet(unsigned x, unsigned y) const { return m_blocksUsed[(y/BLOCK_SIZE)*m_numBlocksX + x/BLOCK_SIZE]; }
        inline bool isBlockSet(unsigned x, unsigned y) const { return m_blocksUsed[y*m_numBlocksX + x]; }

        inline unsigned getWidth() const { return m_width; }
        inline unsigned getHeight() const { return m_height; }
    private:
        unsigned m_width;
        unsigned m_height;
        unsigned m_numBlocksX;
        unsigned m_numBlocksY;
        std::vector<bool> m_blocksUsed;
};

struct ElementHandle {
    unsigned intraBlockIndex : 4;
    unsigned blockIndex : 28;
} __attribute__((packed));

struct MultiplicationWithTransposeRecipe
{
    struct BlockJob {
        unsigned targetBlockIndex;
        std::vector<std::pair<unsigned, unsigned> > sourcePairs;
    };
    std::vector<BlockJob> jobs;
};

struct CholeskyFactorizationRecipe
{
    struct RowElementJob;

    struct DiagonalElementJob {
        unsigned targetBlockIndex;
        unsigned sourceBlockIndex;
        std::vector<unsigned> rowBlocks;
    };

    struct RowElementJob {
        unsigned targetBlockIndex;
        unsigned diagonalHeadBlockIndex;
        unsigned sourceBlockIndex;
        struct Pair {
            unsigned leftBlockIndex;
            unsigned topBlockIndex;
            Pair(unsigned left, unsigned top) : leftBlockIndex(left), topBlockIndex(top) { }
        };
        std::vector<Pair> sumBlocks;
    };

    struct Column {
        std::vector<RowElementJob> rowJobs;
        DiagonalElementJob diagonalJob;
    };

    std::vector<Column> columns;
};

struct MultiplyDiagonalRecipe
{
    std::vector<unsigned> blocks;
};

class Layout
{
    public:
        enum Mode {
            MODE_ARBITRARY,
            MODE_SYMMETRIC,
            MODE_LOWER_TRIANGULAR
        };

        Layout();

        unsigned getBlockIndex(unsigned bx, unsigned by) const;
        ElementHandle getElementHandle(unsigned x, unsigned y) const;

        inline Mode getMode() const { return m_mode; }

        inline unsigned getWidth() const { return m_width; }
        inline unsigned getHeight() const { return m_height; }
        inline unsigned getNumBlocksX() const { return m_numBlocksX; }
        inline unsigned getNumBlocksY() const { return m_numBlocksY; }
        inline unsigned getNumBlocks() const { return m_numBlocks; }

        void debugFindBlockCoord(unsigned blockIndex, unsigned &bx, unsigned &by) const;

        struct BlockEntry {
            uint32_t offset;
            uint32_t blockIndex;

            BlockEntry(uint32_t o, uint32_t b) : offset(o), blockIndex(b) { }
        };
        void drawLayout(RasterImage &image) const;

        inline const std::vector<BlockEntry> &getRow(unsigned row) const { return m_rows[row]; }
        inline const std::vector<BlockEntry> &getCol(unsigned col) const { return m_cols[col]; }

        void createFromGenerator(const LayoutGenerator &generator, bool symmetric = false);
        void createMultiplicationMatrixWithTransposed(const Layout &src, MultiplicationWithTransposeRecipe &recipe);
        void createCholeskyFactorization(const Layout &src, CholeskyFactorizationRecipe &recipe);

        void createMultiplyDiagonalRecipe(MultiplyDiagonalRecipe &recipe);
    private:
        unsigned m_width;
        unsigned m_height;
        unsigned m_numBlocksX;
        unsigned m_numBlocksY;
        unsigned m_numBlocks;

        Mode m_mode;

        std::vector<std::vector<BlockEntry> > m_rows;
        std::vector<std::vector<BlockEntry> > m_cols;
};


class BlockSparseMatrix
{
    public:
        struct Block {
            float values[BLOCK_SIZE*BLOCK_SIZE] __attribute__ ((aligned (16)));
        };

        BlockSparseMatrix();
        ~BlockSparseMatrix();

        void setup(const Layout &layout);

        inline void setValue(const ElementHandle &index, float value) { m_blocks[index.blockIndex].values[index.intraBlockIndex] = value; }
        inline float getValue(const ElementHandle &index) const { return m_blocks[index.blockIndex].values[index.intraBlockIndex]; }

        void performMultiplicationWithTranspose(const BlockSparseMatrix &src, const MultiplicationWithTransposeRecipe &recipe);
        void performCholeskyFactorization(const BlockSparseMatrix &src, const CholeskyFactorizationRecipe &recipe);

        void solveForwardSubstitution(const std::vector<float> &src, std::vector<float> &dst) const;
        void solveBackwardSubstitution(const std::vector<float> &src, std::vector<float> &dst) const;

        void multiplyWithVector(const std::vector<float> &src, std::vector<float> &dst) const;

        void multiplyDiagonal(const MultiplyDiagonalRecipe &recipe, float factor);

        void copyFrom(const BlockSparseMatrix &src);

        void printMatlab() const;

    protected:
        Layout m_layout;
        std::vector<Block> m_blocks;
};

}

/// @}

}


#endif // BLOCKSPARSEMATRIX_H
