/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef LINKEDLIST_HPP_INCLUDED
#define LINKEDLIST_HPP_INCLUDED

#include <assert.h>
#include <stddef.h>


/** @addtogroup Codebase_Group
 *  @{
 */


template<typename HostType>
class LinkedList;

template<typename HostType>
class LinkedListEntry
{
    public:
        typedef LinkedListEntry<HostType> ListEntryType;
        typedef LinkedList<HostType> LinkedListType;

        LinkedListEntry(HostType &host) : m_host(host) {
            m_list = NULL;
            m_prev = m_next = NULL;
        }
        ~LinkedListEntry() {
            removeFromList();
        }

        inline HostType &getHost() { return m_host; }
        inline LinkedListType *getList() { return m_list; }
        inline bool inList() { return m_list != NULL; }
        inline ListEntryType *getPrev() { return m_prev; }
        inline ListEntryType *getNext() { return m_next; }

        void appendToList(LinkedListType *list) {
            removeFromList();

            m_prev = list->m_last;
            m_next = NULL;
            if (list->m_last != NULL)
                list->m_last->m_next = this;
            else
                list->m_first = this;
            list->m_last = this;

            m_list = list;
            m_list->m_count++;

        }
        void removeFromList() {
            if (m_list != NULL) {
                if (m_prev != NULL)
                    m_prev->m_next = m_next;
                else
                    m_list->m_first = m_next;

                if (m_next != NULL)
                    m_next->m_prev = m_prev;
                else
                    m_list->m_last = m_prev;

                m_list->m_count--;
                m_list = NULL;
                m_prev = m_next = NULL;
            }
        }
    private:
        LinkedListEntry(const LinkedListEntry &other) : m_host(other.m_host) { }
        void operator=(const LinkedListEntry &other) { }


        ListEntryType *m_prev, *m_next;
        HostType &m_host;
        LinkedListType *m_list;

};

template<typename HostType>
class LinkedList
{
    public:
        typedef LinkedListEntry<HostType> ListEntryType;

        LinkedList() {
            m_first = m_last = NULL;
            m_count = 0;
        }
        ~LinkedList() {
            assert(m_count == 0);
        }

        inline void operator+=(ListEntryType &listEntry) {
            listEntry.appendToList(this);
        }

        inline void operator-=(ListEntryType &listEntry) {
            listEntry.removeFromList();
        }

        inline ListEntryType *getFirst() { return m_first; }
        inline ListEntryType *getLast() { return m_last; }

        inline unsigned size() const { return m_count; }
    private:
        friend class LinkedListEntry<HostType>;
        ListEntryType *m_first, *m_last;
        unsigned m_count;
};

/// @}

#endif // LINKEDLIST_HPP_INCLUDED
