/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "StackAllocator.h"
#include <stdlib.h>

#include <stdexcept>


StackAllocator::StackAllocator()
{
    addChunk(MIN_CHUNK_SIZE);
    reset();
}

StackAllocator::~StackAllocator()
{
    for (unsigned i = 0; i < m_chunks.size(); i++) {
        free(m_chunks[i].data);
    }
}


void *StackAllocator::allocate(unsigned size)
{
    while (true) {
        Chunk &chunk = m_chunks[m_currentChunk];
        if (m_currentByte+size <= chunk.size) {
            unsigned char *ptr = chunk.data + m_currentByte;
            m_currentByte += size;
            return ptr;
        } else {
            m_currentChunk++;
            m_currentByte = 0;
            if (m_currentChunk == m_chunks.size())
                addChunk(size);
        }
    }
}

void *StackAllocator::allocateAligned16(unsigned size)
{
    m_currentByte = (m_currentByte+0xF) & ~0xF;
    return allocate(size);
}

void StackAllocator::addChunk(unsigned minSize)
{
    unsigned size = (std::max(minSize, 1u)+MIN_CHUNK_SIZE-1) & MIN_CHUNK_SIZE_MASK;
    Chunk chunk;
    if (posix_memalign((void**)&chunk.data, 16, size) != 0)
        throw std::runtime_error("Internal error, out of memory!");
    chunk.size = size;

    m_chunks.push_back(chunk);
}
