/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "BinSFMExporter.h"


#include <SFM/SFM.h>
#include <SFM/Frame.h>
#include <SFM/Track.h>
#include <SFM/TrackObservation.h>
#include <SFM/InternalCameraCalibration.h>

#include "ExtractBasicColorsForTracks.h"

#include <fstream>

namespace SFM {
namespace Utilities {

namespace detail {
struct BinaryFormatterForBraindeadFstream
{
    public:
        BinaryFormatterForBraindeadFstream(std::fstream &s) : stream(s) { }
        template<typename type>
        BinaryFormatterForBraindeadFstream &operator<<(const type &var) {
            stream.write((const char*)&var, sizeof(type));
            return *this;
        }
        void blockWrite(const void *ptr, size_t size) {
            stream.write((const char*)ptr, size);
        }

        BinaryFormatterForBraindeadFstream &operator<<(const std::string &var) {
            uint16_t len = var.length();
            stream << len;
            stream.write(var.c_str(), len);
            return *this;
        }
    private:
        std::fstream &stream;
};

struct BinTrack {
    LinAlg::Vector4f position;
    uint32_t color;
};
}

void exportBinSFM(const boost::filesystem::path &destinationPath, SFM &sfm)
{
    std::vector<LinAlg::Vector3f> colors;
    extractBasicColorsForTracks(sfm, colors);

    const unsigned version = 1;
    const unsigned magic = 'S' | ('S' << 8) | ('F' << 16) | ('M' << 24);

    std::fstream file;
    file.exceptions(std::ifstream::failbit | std::ifstream::badbit);

    file.open(destinationPath.native().c_str(), std::fstream::binary | std::fstream::out);
    detail::BinaryFormatterForBraindeadFstream stream(file);

    stream << magic << version;

    uint32_t numCameras = sfm.getFrames().size();
    stream << numCameras;
    for (unsigned i = 0; i < sfm.getFrames().size(); i++) {
        stream << sfm.getFrames()[i]->getImageFilename();
        const LinAlg::Matrix4x4f &PV = sfm.getFrames()[i]->getCamera().getProjectionViewMatrix();
        stream.blockWrite(&PV[0][0], 4*4*sizeof(float));
    }

    std::vector<detail::BinTrack> binTracks;
    binTracks.reserve(sfm.getTrackList().reservedSize());
    for (const Track &track : sfm.getTrackList()) {
        if (track.getState() == Track::STATE_DISABLED)
            continue;

        detail::BinTrack binTrack;
        binTrack.position = track.getLastWSPositionEstimate();
        LinAlg::Vector3f c = colors[sfm.getTrackList().findIndexOf(&track)];
        binTrack.color = (std::min<int>(255, c[0] * 256) << 0) |
                         (std::min<int>(255, c[1] * 256) << 8) |
                         (std::min<int>(255, c[2] * 256) << 16) |
                         (0xFF << 24);

        binTracks.push_back(binTrack);
    }

    uint64_t numTracks = binTracks.size();
    stream << numTracks;
    stream.blockWrite(&binTracks[0], binTracks.size() * sizeof(detail::BinTrack));

    file.close();

}

}
}

