/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef CONFIGURATIONREFLECTION_H
#define CONFIGURATIONREFLECTION_H

#include <string>
#include <boost/lexical_cast.hpp>
#include <boost/filesystem.hpp>
#include <limits>

namespace SFM {

namespace config {
struct BundleAdjustmentParameterConfig;
struct BundleAdjustmentStructureConfig;
struct CudaConfig;
struct FeatureExtractionConfig;
struct FeatureMatchingConfig;
struct SubsetSelectionConfig;
struct SfMConfig;
}


namespace Utilities {

/** @addtogroup SFMUtilities_Group
 *  @{
 */

/**
 * @brief Allows a frontend application to reflect on the parameters of the SFM framework.
 * @details This allows the frontends to display and set parameters in an automatic fashion.
 * If all frontends use this reflection mechanism then new parameters only need to described here,
 * as opposed to once for every frontend.
 */
class ConfigurationReflection
{
    public:

        class BaseElement {
            public:
                BaseElement(const char *name, const char *description) : m_name(name), m_description(description) { }
                virtual ~BaseElement() { }

                inline const char *getName() const { return m_name; }
                inline const char *getDescription() const { return m_description; }
            protected:
                const char *m_name;
                const char *m_description;
        };

        class BaseParameter : public BaseElement {
            public:
                BaseParameter(const char *name, const char *description) : BaseElement(name, description) { }

                virtual std::string getValueAsStr() const = 0;
                virtual bool parseValueFromString(const std::string &str) = 0;
        };


        class NumericParameter : public BaseParameter {
            public:
                NumericParameter(const char *name, const char *description) : BaseParameter(name, description) { }

                virtual std::string getMinValueAsStr() const = 0;
                virtual std::string getMaxValueAsStr() const = 0;
        };

        template<typename Type>
        class NumericParameterTmpl : public NumericParameter {
            public:
                NumericParameterTmpl(const char *name, const char *description, Type *value,
                                     const Type &minValue = std::numeric_limits<Type>::min(), const Type &maxValue = std::numeric_limits<Type>::max())
                    : NumericParameter(name, description), m_min(minValue), m_max(maxValue), m_value(value) { }

                std::string getValueAsStr() const override {
                    return boost::lexical_cast<std::string>(*m_value);
                }
                bool parseValueFromString(const std::string &str) override {
                    try {
                        *m_value = boost::lexical_cast<Type>(str);
                    } catch (const boost::bad_lexical_cast &) {
                        return false;
                    }
                    return true;
                }

                std::string getMinValueAsStr() const override {
                    return boost::lexical_cast<std::string>(m_min);
                }
                std::string getMaxValueAsStr() const override {
                    return boost::lexical_cast<std::string>(m_max);
                }

                inline const Type &getMinValue() const { return m_min; }
                inline const Type &getMaxValue() const { return m_max; }
                inline const Type &getValue() const { return *m_value; }
                inline Type &getValue() { return *m_value; }
            protected:
                Type m_min;
                Type m_max;
                Type *m_value;
        };

        typedef NumericParameterTmpl<float> FloatParameter;
        typedef NumericParameterTmpl<int> IntParameter;
        typedef NumericParameterTmpl<unsigned> UnsignedParameter;

        class StringParameter : public BaseParameter {
            public:
                StringParameter(const char *name, const char *description, std::string *value)
                    : BaseParameter(name, description), m_value(value) { }

                std::string getValueAsStr() const override {
                    return *m_value;
                }
                bool parseValueFromString(const std::string &str) override {
                    *m_value = str;
                    return true;
                }
            protected:
                std::string *m_value;
        };

        class BaseEnumParameter : public BaseParameter {
            public:
                BaseEnumParameter(const char *name, const char *description,
                                  unsigned numChoices, const char **choiceNames, const char **choiceDescriptions) :
                                      BaseParameter(name, description), m_numChoices(numChoices),
                                        m_choiceNames(choiceNames), m_choiceDescriptions(choiceDescriptions) { }

                unsigned getNumChoices() const { return m_numChoices; }
                const char *getChoiceName(unsigned index) const { return m_choiceNames[index]; }
                const char *getChoiceDescription(unsigned index) const { return m_choiceDescriptions[index]; }
            protected:
                unsigned m_numChoices;
                const char **m_choiceNames;
                const char **m_choiceDescriptions;
        };

        template<typename Type>
        class EnumParameterTmpl : public BaseEnumParameter {
            public:
                EnumParameterTmpl(const char *name, const char *description, Type *value,
                                     unsigned numChoices, const char *choiceNames[], const char *choiceDescriptions[])
                    : BaseEnumParameter(name, description, numChoices, choiceNames, choiceDescriptions), m_value(value) { }

                std::string getValueAsStr() const override {
                    return m_choiceNames[(unsigned)*m_value];
                }
                bool parseValueFromString(const std::string &str) override {
                    for (unsigned i = 0; i < m_numChoices; i++)
                        if (str == m_choiceNames[i]) {
                            *m_value = (Type)i;
                            return true;
                        }
                    return false;
                }
            protected:
                Type *m_value;
        };

        class ParameterGroup : public BaseElement {
            public:
                ParameterGroup(const char *name, const char *description) : BaseElement(name, description) { }

                ParameterGroup &operator<<(BaseElement *element) {
                    m_elements.push_back(std::unique_ptr<BaseElement>(element));
                    return *this;
                }

                class iterator {
                    public:
                        iterator(ParameterGroup &grp, unsigned index) : m_index(index), m_grp(grp) { }

                        bool operator!=(const iterator &other) const {
                            return (m_index != other.m_index) || (&m_grp != &other.m_grp);
                        }

                        inline BaseElement &operator*() {
                            return *m_grp.m_elements[m_index];
                        }
                        inline BaseElement *operator->() {
                            return m_grp.m_elements[m_index].get();
                        }

                        iterator operator++() {
                            m_index++;
                            return *this;
                        }
                    private:
                        unsigned m_index;
                        ParameterGroup &m_grp;
                };

                iterator begin() {
                    return iterator(*this, 0);
                }

                iterator end() {
                    return iterator(*this, m_elements.size());
                }
            protected:
                friend class iterator;
                std::vector<std::unique_ptr<BaseElement>> m_elements;
        };



        ConfigurationReflection(config::BundleAdjustmentParameterConfig &config);
        ConfigurationReflection(config::BundleAdjustmentStructureConfig &config);
        ConfigurationReflection(config::CudaConfig &config);
        ConfigurationReflection(config::FeatureExtractionConfig &config);
        ConfigurationReflection(config::FeatureMatchingConfig &config);
        ConfigurationReflection(config::SubsetSelectionConfig &config);
        ConfigurationReflection(config::SfMConfig &config);

        ParameterGroup &getReflection() { return m_root; }
    protected:
        ParameterGroup m_root;
};
/// @}

}
}

#endif // CONFIGURATIONREFLECTION_H
