/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "ImageResampler.h"

#include <cudaUtilities/CudaKernel.h>

#include <cudaUtilities/CudaTextureMemory.h>

#include <cudaUtilities/CudaTextureReference.h>
#include <cudaUtilities/CudaSurfaceReference.h>

#include <cudaKernels/ImageResampling.cuh>

#include <tools/RasterImage.h>


#include <fcntl.h>
#include <sys/sendfile.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

namespace SFM {

namespace Utilities {

ImageResampler::ImageResampler(const config::CudaConfig &cudaConfig)
{
    m_codeModule.loadFromFile((cudaConfig.kernelFatbinBaseDir +"ImageResampling.fatbin").c_str());

    m_kernelRadialPolynomial234 = std::unique_ptr<CudaUtils::CudaKernel>(m_codeModule.getKernel("undistortRadialPolynomial234"));
    m_kernelHomography = std::unique_ptr<CudaUtils::CudaKernel>(m_codeModule.getKernel("homographyImageResampling"));
    m_inputTexRef = std::unique_ptr<CudaUtils::CudaTextureReference>(m_codeModule.getTexReference("source"));
    m_outputSurfRef = std::unique_ptr<CudaUtils::CudaSurfaceReference>(m_codeModule.getSurfReference("destination"));

    m_inputTexRef->setTexelFilterMode(CudaUtils::CudaTextureReference::FILTER_MODE_LINEAR);
    m_inputTexRef->setMinMaxMipLevel(0, 0);
    m_inputTexRef->setCoordinateNormalization(false);
}

void ImageResampler::resampleHomography(const std::vector<std::pair<std::string, std::string> > &filenames,
                     const LinAlg::Matrix4x4f &homography,
                     const ProgressReportCallback &progressReportCallback)
{
    throw std::runtime_error("Not implemented yet!");
    /*
    m_sourceImageTexRef->bindMipmappedTexture(&src);
    m_sourceImageTexRef->setMinMaxMipLevel(0, src.getNumLevel()-1);

    m_outputArray.resize(dst.getWidth() * dst.getHeight() * 4);


    HomographyImageResamplingKernelParams kernelParams;
    kernelParams.srcWidth = src.getLevel(0).getWidth();
    kernelParams.srcHeight = src.getLevel(0).getHeight();
    kernelParams.width = dst.getWidth();
    kernelParams.height = dst.getHeight();
    kernelParams.LODbias = lodBias;
    kernelParams.dstRGBA = (unsigned*) m_outputArray.getPtr();

    LinAlg::Matrix3x3f hom = homography * LinAlg::Scale2D(LinAlg::Fill(1.0f / dst.getWidth(), 1.0f / dst.getHeight()));
    for (unsigned i = 0; i < 3; i++)
        for (unsigned j = 0; j < 3; j++)
            kernelParams.homography[i*3+j] = hom[i][j];

    m_kernel->launch(LinAlg::Fill(16u, 16u, 1u),
                        LinAlg::Fill((kernelParams.width+15u)/16u, (kernelParams.height+15u)/16u, 1u),
                        &kernelParams, sizeof(kernelParams));

    m_outputArray.download(dst.getData(), dst.getWidth() * dst.getHeight() * 4);
    */
}


void ImageResampler::undistortImages(const std::vector<std::pair<std::string, std::string> > &filenames,
                     const SFM::BundleAdjustment::RadialDistortionParametrization &distortion,
                     const ProgressReportCallback &progressReportCallback)
{

    switch (distortion.type) {
        case config::BundleAdjustmentStructureConfig::RadialDistortionType::NoRadialDistortion:
            copyImages(filenames, progressReportCallback);
        break;
        case config::BundleAdjustmentStructureConfig::RadialDistortionType::Polynomial_234:
            undistortImagesRadialPolynomial234(filenames, distortion.polynomial234.kappa, progressReportCallback);
        break;
        default:
            throw std::runtime_error("Unsupported distortion type!");
    }
}


// found on stack overflow. Nice :-)
void ImageResampler::copyFile(const char *dst, const char *src)
{
    int read_fd;
    int write_fd;
    struct stat stat_buf;
    off_t offset = 0;

    /* Open the input file. */
    read_fd = open (src, O_RDONLY);
    if (read_fd == -1) {
        std::stringstream msg;
        msg << "Could not copy file " << src << ": Source file could not be opened!";
        throw std::runtime_error(msg.str());
    }
    /* Stat the input file to obtain its size. */
    fstat (read_fd, &stat_buf);
    /* Open the output file for writing, with the same permissions as the
    source file. */
    write_fd = creat (dst, stat_buf.st_mode);
    if (write_fd == -1) {
        std::stringstream msg;
        msg << "Could not copy file " << src << ": Destination file could not be opened!";
        throw std::runtime_error(msg.str());
    }
    /* Blast the bytes from one file to the other. */
    int sendFileRet = sendfile (write_fd, read_fd, &offset, stat_buf.st_size);
    if (sendFileRet == -1) {
        std::stringstream msg;
        msg << "Could not copy file " << src << ": An error occured while copying!";
        throw std::runtime_error(msg.str());
    }
    /* Close up. */
    close (read_fd);
    close (write_fd);
}

void ImageResampler::copyImages(const std::vector<std::pair<std::string, std::string> > &filenames,
                                const ProgressReportCallback &progressReportCallback)
{
    ProgressReport progressReport;
    progressReport.totalNumImages = filenames.size();

    for (unsigned i = 0; i < filenames.size(); i++) {
        progressReport.currentImage = i;
        if (progressReportCallback)
            progressReportCallback(progressReport);

        copyFile(filenames[i].second.c_str(), filenames[i].first.c_str());
    }
}

void ImageResampler::undistortImagesRadialPolynomial234(const std::vector<std::pair<std::string, std::string> > &filenames,
                                                        const float *kappa, const ProgressReportCallback &progressReportCallback)
{
    RasterImage sourceImage;
    RasterImage destinationImage;

    ProgressReport progressReport;
    progressReport.totalNumImages = filenames.size();

    for (unsigned i = 0; i < filenames.size(); i++) {
        progressReport.currentImage = i;
        if (progressReportCallback)
            progressReportCallback(progressReport);

        sourceImage.loadFromFile(filenames[i].first.c_str());

        CudaUtils::CudaTextureMemory gpuSourceImage;
        gpuSourceImage.resize(sourceImage.getWidth(), sourceImage.getHeight(), 0, CU_AD_FORMAT_UNSIGNED_INT8, 4, 0);
        gpuSourceImage.syncUploadAll(sourceImage.getData(), 4*sourceImage.getWidth());

        CudaUtils::CudaTextureMemory gpuOutputImage;
        gpuOutputImage.resize(sourceImage.getWidth(), sourceImage.getHeight(), 0, CU_AD_FORMAT_UNSIGNED_INT8, 4, CUDA_ARRAY3D_SURFACE_LDST);


        {
            m_inputTexRef->bindTexture(&gpuSourceImage);
            m_outputSurfRef->bindTexture(&gpuOutputImage);

            UndistortRadialPolynomial234KernelParams params;
            params.width = sourceImage.getWidth();
            params.height = sourceImage.getHeight();
            params.imageCenterX = params.width * 0.5f + 0.5f;
            params.imageCenterY = params.height * 0.5f + 0.5f;
            params.rcpFac = 2.0f / sourceImage.getWidth();
            params.fac = 1.0f / params.rcpFac;
            params.kappa[0] = kappa[0];
            params.kappa[1] = kappa[1];
            params.kappa[2] = kappa[2];

            m_kernelRadialPolynomial234->launch(LinAlg::Fill(16u, 16u, 1u),
                                        LinAlg::Fill((sourceImage.getWidth()+15u)/16u, (sourceImage.getHeight()+15u)/16u, 1u),
                                        &params, sizeof(params));

        }

        destinationImage.resize(sourceImage.getWidth(), sourceImage.getHeight());

        gpuOutputImage.syncDownloadAll(destinationImage.getData(), 4*destinationImage.getWidth());
        destinationImage.writeToFile(filenames[i].second.c_str(), 100);
    }
}

}
}
