/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "MinimumCutAnalysis.h"

#include <SFM/SFM.h>
#include <SFM/Track.h>
#include <SFM/Frame.h>
#include <SFM/InternalCameraCalibration.h>

#include <map>

// for min cut:
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/graph_traits.hpp>
#include <boost/graph/one_bit_color_map.hpp>
#include <boost/graph/stoer_wagner_min_cut.hpp>
#include <boost/property_map/property_map.hpp>
#include <boost/typeof/typeof.hpp>


namespace SFM {
namespace Utilities {


struct edge_t
{
    unsigned first;
    unsigned second;
};


CameraGraphMinimumCut analyzeMinCut(SFM &sfm, bool onlySubset)
{
    typedef boost::adjacency_list<boost::vecS, boost::vecS, boost::undirectedS,
            boost::no_property, boost::property<boost::edge_weight_t, int> > undirected_graph;
    typedef boost::property_map<undirected_graph, boost::edge_weight_t>::type weight_map_type;
    typedef boost::property_traits<weight_map_type>::value_type weight_type;


    std::vector<edge_t> edges;
    std::vector<weight_type> weights;

    std::map<std::pair<unsigned, unsigned>, float> edgeMap;

    for (const Track &track : sfm.getTrackList()) {
        if (track.getState() == Track::STATE_DISABLED) continue;
        if (onlySubset && (track.getState() == Track::STATE_SEMI_ACTIVE)) continue;


        for (unsigned k = 0; k < track.getObservations().reservedSize(); k++) {
            if (!track.getObservations().inUse(k)) continue;
            const TrackObservation &obs1 = track.getObservations()[k];
            if (obs1.isFaulty())
                continue;

            unsigned obs1FrameIndex = obs1.getFrame()->getIndex();

            for (unsigned l = k+1; l < track.getObservations().reservedSize(); l++) {
                if (!track.getObservations().inUse(l)) continue;
                const TrackObservation &obs2 = track.getObservations()[l];
                if (obs2.isFaulty())
                    continue;

                unsigned obs2FrameIndex = obs2.getFrame()->getIndex();

                //assert(obs1FrameIndex != obs2FrameIndex);

                if (obs1FrameIndex == obs2FrameIndex) continue;

                auto key = (obs1FrameIndex < obs2FrameIndex)?
                                std::pair<unsigned, unsigned>(obs1FrameIndex, obs2FrameIndex):
                                std::pair<unsigned, unsigned>(obs2FrameIndex, obs1FrameIndex);

                edgeMap[key]+=1.0f;
            }
        }
    }

    int totalCapacity = 0;
    edges.reserve(edgeMap.size());
    weights.reserve(edgeMap.size());
    for (auto it = edgeMap.begin(); it != edgeMap.end(); ++it) {
        edge_t e;
        e.first = it->first.first;
        e.second = it->first.second;
        edges.push_back(e);
        weights.push_back(it->second);
        totalCapacity += it->second;
    }


    undirected_graph g(edges.begin(), edges.end(), &weights[0], sfm.getFrames().size(), edges.size());

    auto parities = boost::make_one_bit_color_map(num_vertices(g), get(boost::vertex_index, g));
    int w = boost::stoer_wagner_min_cut(g, get(boost::edge_weight, g), boost::parity_map(parities));

    CameraGraphMinimumCut result;
    result.minimumCutCapacity = w;
    result.totalCapacity = totalCapacity;

    for (unsigned i = 0; i < num_vertices(g); ++i) {
        if (get(parities, i))
            result.set1.push_back(i);
        else
            result.set2.push_back(i);
    }
/*
    std::cout << "The minimum cut capacity is " << w << " of a total capacity of " << totalCapacity << std::endl;
    std::cout << "     " << w*100.0f/totalCapacity << " %" << std::endl;
    std::cout << "One set of vertices consists of:" << std::endl;
    for (unsigned i = 0; i < num_vertices(g); ++i) {
        if (get(parities, i))
            std::cout << i << std::endl;
    }
    std::cout << std::endl;

    std::cout << "The other set of vertices consists of:" << std::endl;
    for (unsigned i = 0; i < num_vertices(g); ++i) {
        if (!get(parities, i))
            std::cout << i << std::endl;
    }
    std::cout << std::endl;
*/
    return result;
}



}
}
