/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "XmlExporter.h"


#include <SFM/SFM.h>
#include <SFM/Frame.h>
#include <SFM/InternalCameraCalibration.h>


#include <tinyxml.h>

namespace SFM {
namespace Utilities {

void exportXml(const boost::filesystem::path &destinationPath, SFM &sfm)
{
    TiXmlDocument document;
    TiXmlElement *root = new TiXmlElement("scan");
    document.LinkEndChild(root);
    TiXmlElement *clip = new TiXmlElement("video_clip");
    root->LinkEndChild(clip);

    for (unsigned i = 0; i < sfm.getFrames().size(); i++) {
        Frame &frame = *sfm.getFrames()[i];

        LinAlg::Matrix3x4f projectionMatrix = frame.getCamera().getProjectionViewMatrix().dropRow(2);

        TiXmlElement *frameNode = new TiXmlElement("frame");

        {
            TiXmlElement *node = new TiXmlElement("image");
            node->SetAttribute("filename", frame.getImageFilename().c_str());
            node->SetAttribute("width", frame.getWidth());
            node->SetAttribute("height", frame.getHeight());
            frameNode->LinkEndChild(node);
        }
        {
            TiXmlElement *node = new TiXmlElement("pose");

            std::stringstream stream;
            for (unsigned i = 0; i < 3; i++)
                for (unsigned j = 0; j < 4; j++)
                    stream << std::setprecision(15) << projectionMatrix[i][j] << " ";

            node->SetAttribute("projection_matrix", stream.str());
            frameNode->LinkEndChild(node);
        }
        const BundleAdjustment::RadialDistortionParametrization &radialDistortion = frame.getCamera().getInternalCalibration()->getRadialDistortion();
        switch (radialDistortion.type) {
            case config::BundleAdjustmentStructureConfig::RadialDistortionType::NoRadialDistortion:
            break;
            case config::BundleAdjustmentStructureConfig::RadialDistortionType::Polynomial_234:
                {
                    TiXmlElement *node = new TiXmlElement("distortion");

                    node->SetAttribute("type", "Polynomial_234");

                    std::stringstream stream;
                    for (unsigned i = 0; i < 3; i++)
                        stream << std::setprecision(15) << radialDistortion.polynomial234.kappa[i] << " ";

                    node->SetAttribute("kappa", stream.str());
                    frameNode->LinkEndChild(node);
                }
            break;
            default:
                throw std::runtime_error("Unhandled radial distortion type!");
        };

        clip->LinkEndChild(frameNode);
    }
    if (!document.SaveFile(destinationPath.native().c_str()))
        throw std::runtime_error("Could not write XML file!");
}

}
}

