/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/**

@page Page_SparseReconstructionViewer The sparse reconstruction viewer

@section Page_SparseReconstructionViewer_Intro Introduction

The sparse reconstruction viewer is a small program to inspect the sparse reconstruction saved in a .binSfm
file. The program opens a small OpenGL window and displays the cameras as wireframes and the tracks as points.

@section Page_SparseReconstructionViewer_Build How to Build the Framework

In addition to a Code::Blocks project file, the framework comes with a set of Makefiles for
the most important parts.

The following builds the release build of the sparse reconstruction viewer.
@code{.unparsed}
$ cd code/SparseReconstructionViewer
$ make release -j 8
@endcode

@section Page_SparseReconstructionViewer_Usage Usage

@subsection Page_CommandLineFrontend_Usage_Example Example
The following loads the sparse reconstruction of the volcano dataset. Using primusrun (or optirun)
is only needed when the GPU is subject to NVidia Optimus (energy saving mechanism on notebooks).
@code{.unparsed}
$ cd code/SparseReconstructionViewer
$ primusrun bin/Release/SparseReconstructionViewer ../../results/datasets/Volcano/new+radial/sparseReconstruction.binSFM
@endcode

@subsection Page_SparseReconstructionViewer_Usage_Controls Controls

As long as the sparse reconstruction viewer has the focus, the following controls can be used:

<TABLE>
<TR> <TH>Control</TH> <TH>Effect</TH></TR>
<TR> <TD>Hold left mouse button and drag left/right</TD> <TD>pan camera left/right</TD></TR>
<TR> <TD>Hold left mouse button and drag up/down</TD> <TD>tilt camera up/down</TD></TR>
<TR> <TD>W</TD> <TD>move camera forward</TD></TR>
<TR> <TD>S</TD> <TD>move camera backward</TD></TR>
<TR> <TD>A</TD> <TD>move camera left</TD></TR>
<TR> <TD>D</TD> <TD>move camera right</TD></TR>
<TR> <TD>C</TD> <TD>move camera down</TD></TR>
<TR> <TD>SPACE</TD> <TD>move camera up</TD></TR>
<TR> <TD>+</TD> <TD>increase the size of the wireframe cameras</TD></TR>
<TR> <TD>-</TD> <TD>decrease the size of the wireframe cameras</TD></TR>
<TR> <TD>r</TD> <TD>render the current view into a file called 'test.svg' with the wireframes as vector graphics and the point cloud as a raster image</TD></TR>
</TABLE>

 */





#include <iostream>

#include <platform/X11Window.h>
#include <graphics/RenderDevice.h>

#include <GL/glew.h>

#include <tools/SvgComposer.h>
#include <tools/LinAlg.h>
#include <tools/PCVToolbox.hpp>

#include <tools/SpectatorCamera.h>
#include <fstream>
#include <graphics/BufferObject.h>
#include <algorithm>

struct BinaryFormatterForBraindeadFstream
{
    public:
        BinaryFormatterForBraindeadFstream(std::fstream &s) : stream(s) { }
        template<typename type>
        BinaryFormatterForBraindeadFstream &operator>>(type &var) {
            stream.read((char*)&var, sizeof(type));
            return *this;
        }
        void blockRead(void *ptr, size_t size) {
            stream.read((char*)ptr, size);
        }

        BinaryFormatterForBraindeadFstream &operator>>(std::string &var) {
            uint16_t len;
            stream >> len;
            var.resize(len);
            stream.read((char*)&var[0], len);
            return *this;
        }
    private:
        std::fstream &stream;
};


class SparseReconstruction
{
    public:
        struct Track {
            LinAlg::Vector4f position;
            uint32_t color;
        };
        struct Camera {
            std::string filename;
            LinAlg::Matrix4x4f PV;
            LinAlg::Matrix4x4f invPV;
        };

        void load(const std::string &filename);

        void storeAsPlyPointCloud(const std::string &filename);

        inline const std::vector<Camera> &getCameras() const { return m_cameras; }
        inline const std::vector<Track> &getTracks() const { return m_tracks; }


        void rasterPointImage(RasterImage &image, unsigned width, unsigned height, float pointSize, const LinAlg::Matrix4x4f &PV);

        void drawSVGImage(const std::string &filename, unsigned width, unsigned height, float pointSize, const LinAlg::Matrix4x4f &PV,
                          float cameraSize, float cameraAspectRatio);
    private:
        std::vector<Camera> m_cameras;
        std::vector<Track> m_tracks;
};

void SparseReconstruction::load(const std::string &filename)
{
    m_cameras.clear();
    m_tracks.clear();

    std::fstream file;
    file.exceptions(std::ifstream::failbit | std::ifstream::badbit);

    file.open(filename, std::fstream::binary | std::fstream::in);
    BinaryFormatterForBraindeadFstream stream(file);

    unsigned magic, version;
    stream >> magic >> version;
    // todo: check

    uint32_t numCameras;
    stream >> numCameras;
    m_cameras.resize(numCameras);
    for (unsigned i = 0; i < m_cameras.size(); i++) {
        if (version >= 1)
            stream >> m_cameras[i].filename;
        stream.blockRead(&m_cameras[i].PV[0][0], 4*4*sizeof(float));
        m_cameras[i].invPV = m_cameras[i].PV;
        m_cameras[i].invPV.GaussJordanInvert();
    }

    uint64_t numTracks;
    stream >> numTracks;
    m_tracks.resize(numTracks);
    stream.blockRead(&m_tracks[0], numTracks * sizeof(Track));
}


void SparseReconstruction::storeAsPlyPointCloud(const std::string &filename)
{
    std::fstream file;
    file.exceptions(std::ifstream::failbit | std::ifstream::badbit);

    file.open(filename, std::fstream::out);

    file << "ply" << std::endl
         << "format ascii 1.0" << std::endl
         << "element vertex " << m_tracks.size() << std::endl
         << "property float x" << std::endl
         << "property float y" << std::endl
         << "property float z" << std::endl
         << "property uchar diffuse_red" << std::endl
         << "property uchar diffuse_green" << std::endl
         << "property uchar diffuse_blue" << std::endl
         << "end_header" << std::endl;

    for (unsigned i = 0; i < m_tracks.size(); i++) {
        LinAlg::Vector3f pos = m_tracks[i].position.StripHom() / m_tracks[i].position[3];
        file << pos[0] << " " << pos[1] << " " << pos[2] << " "
             << ((m_tracks[i].color >> 0) & 0xFF) << " " << ((m_tracks[i].color >> 8) & 0xFF) << " " << ((m_tracks[i].color >> 16) & 0xFF) << std::endl;
    }
}


struct ProjectedTrack
{
    LinAlg::Vector3f clipSpacePos;
    LinAlg::Vector2f screenCenter;
    uint32_t color;
    bool operator<(const ProjectedTrack &other) const {
        return clipSpacePos[2] > other.clipSpacePos[2];
    }
};

struct Tile
{
    std::vector<ProjectedTrack> tracks;
};

void SparseReconstruction::rasterPointImage(RasterImage &image, unsigned width, unsigned height, float pointSize, const LinAlg::Matrix4x4f &PV)
{
    std::vector<Tile> tiles;

    const unsigned tileSize = 16;
    const unsigned tileRows = (height + tileSize-1) / tileSize;
    const unsigned tileColumns = (width + tileSize-1) / tileSize;

    tiles.resize(tileRows*tileColumns);

    for (unsigned i = 0; i < m_tracks.size(); i++) {
        ProjectedTrack pt;
        LinAlg::Vector4f P = PV * m_tracks[i].position;

        if (std::abs(P[3]) < 1e-20f)
            continue;

        pt.clipSpacePos = P.StripHom() / P[3];

        if ((pt.clipSpacePos[2] < -1.0f) || (pt.clipSpacePos[2] > 1.0f))
            continue;

        pt.screenCenter[0] = pt.clipSpacePos[0] * (width*0.5f) + (width*0.5f);
        pt.screenCenter[1] = -pt.clipSpacePos[1] * (height*0.5f) + (height*0.5f);

        if (((pt.screenCenter[0] - pointSize) < 0.0f) ||
            ((pt.screenCenter[1] - pointSize) < 0.0f) ||
            ((pt.screenCenter[0] + pointSize) > width) ||
            ((pt.screenCenter[1] + pointSize) > height))
                continue;

        pt.color = m_tracks[i].color;

        int tileX1 = (pt.screenCenter[0] - pointSize) / tileSize;
        int tileX2 = (pt.screenCenter[0] + pointSize) / tileSize;

        int tileY1 = (pt.screenCenter[1] - pointSize) / tileSize;
        int tileY2 = (pt.screenCenter[1] + pointSize) / tileSize;

        tileX1 = std::min<int>(std::max<int>(tileX1, 0), tileColumns-1);
        tileX2 = std::min<int>(std::max<int>(tileX2, 0), tileColumns-1);

        tileY1 = std::min<int>(std::max<int>(tileY1, 0), tileRows-1);
        tileY2 = std::min<int>(std::max<int>(tileY2, 0), tileRows-1);

        for (int k = tileY1; k <= tileY2; k++)
            for (int l = tileX1; l <= tileX2; l++) {
                Tile &tile = tiles[k*tileColumns + l];
                tile.tracks.push_back(pt);
            }
    }

    const float rcpSqrPointSize = 1.0f / (pointSize*pointSize);
    image.resize(width, height);
    for (unsigned k = 0; k < tileRows; k++)
        for (unsigned l = 0; l < tileColumns; l++) {
            Tile &tile = tiles[k*tileColumns + l];
            std::sort(tile.tracks.begin(),  tile.tracks.end());

            for (unsigned y = 0; y < tileSize; y++) {
                for (unsigned x = 0; x < tileSize; x++) {

                    int posX = x+l*tileSize;
                    int posY = y+k*tileSize;

                    float r = 255.0f;
                    float g = 255.0f;
                    float b = 255.0f;

                    for (unsigned i = 0; i < tile.tracks.size(); i++) {
                        float dX = posX - tile.tracks[i].screenCenter[0];
                        float dY = posY - tile.tracks[i].screenCenter[1];

                        float sqrD = dX*dX + dY*dY;

                        float alpha = std::max(0.0f, 1.0f - sqrD * rcpSqrPointSize);

                        r = r * (1.0f - alpha) + ((tile.tracks[i].color >> 0) & 0xFF) * alpha;
                        g = g * (1.0f - alpha) + ((tile.tracks[i].color >> 8) & 0xFF) * alpha;
                        b = b * (1.0f - alpha) + ((tile.tracks[i].color >> 16) & 0xFF) * alpha;
                    }

                    uint32_t color = (std::min<int>(r, 255) << 0) |
                                     (std::min<int>(g, 255) << 8) |
                                     (std::min<int>(b, 255) << 16) |
                                     (0xFF << 24);

                    if ((posY < height) && (posX < width))
                        image.getData()[posY * image.getWidth() + posX] = color;
                }
            }
        }
}

void projectLineToSvg(SvgComposer &svgcomposer, const LinAlg::Matrix4x4f &PV, unsigned width, unsigned height,
                      const LinAlg::Vector4f X1, const LinAlg::Vector4f X2)
{
    LinAlg::Vector4f Y1 = PV * X1;
    LinAlg::Vector4f Y2 = PV * X2;

    if ((std::abs(Y1[3]) < 1e-20f) || (std::abs(Y2[3]) < 1e-20f))
        return;

    Y1 /= Y1[3];
    Y2 /= Y2[3];

    if ((std::abs(Y1[2]) < -1.0f) || (std::abs(Y2[2]) < -1.0f) ||
        (std::abs(Y1[2]) > 1.0f) || (std::abs(Y2[2]) > 1.0f))
        return;


    Y1[0] = Y1[0] * (width*0.5f) + (width*0.5f);
    Y1[1] = -Y1[1] * (height*0.5f) + (height*0.5f);

    Y2[0] = Y2[0] * (width*0.5f) + (width*0.5f);
    Y2[1] = -Y2[1] * (height*0.5f) + (height*0.5f);

    svgcomposer << SvgComposer::Line(Y1[0], Y1[1], Y2[0], Y2[1]).setStroke("rgb(0,0,0)").setStrokeWidth(2.0f);

}


void SparseReconstruction::drawSVGImage(const std::string &filename, unsigned width, unsigned height, float pointSize, const LinAlg::Matrix4x4f &PV,
                          float cameraSize, float cameraAspectRatio)
{
    std::fstream file;
    file.exceptions(std::ifstream::failbit | std::ifstream::badbit);
    file.open(filename, std::fstream::out);

    SvgComposer svgComposer(file, width, height);

    RasterImage bgImage;
    rasterPointImage(bgImage, width, height, pointSize, PV);

    svgComposer << SvgComposer::Image(bgImage).setJpgQuality(100);

    const float near = cameraSize;
    const float far = cameraSize / 10.0f;
    for (unsigned i = 0; i < m_cameras.size(); i++) {
        const LinAlg::Matrix4x4f &invPV = m_cameras[i].invPV;
        projectLineToSvg(svgComposer, PV, width, height,
            invPV*LinAlg::Fill(-1.0f, -cameraAspectRatio, near, 1.0f), invPV*LinAlg::Fill(1.0f, -cameraAspectRatio, near, 1.0f));

        projectLineToSvg(svgComposer, PV, width, height,
            invPV*LinAlg::Fill(1.0f, -cameraAspectRatio, near, 1.0f), invPV*LinAlg::Fill(1.0f, cameraAspectRatio, near, 1.0f));

        projectLineToSvg(svgComposer, PV, width, height,
            invPV*LinAlg::Fill(1.0f, cameraAspectRatio, near, 1.0f), invPV*LinAlg::Fill(-1.0f, cameraAspectRatio, near, 1.0f));

        projectLineToSvg(svgComposer, PV, width, height,
            invPV*LinAlg::Fill(-1.0f, cameraAspectRatio, near, 1.0f), invPV*LinAlg::Fill(-1.0f, -cameraAspectRatio, near, 1.0f));


        projectLineToSvg(svgComposer, PV, width, height,
            invPV*LinAlg::Fill(-1.0f, -cameraAspectRatio, far, 1.0f), invPV*LinAlg::Fill(1.0f, -cameraAspectRatio, far, 1.0f));

        projectLineToSvg(svgComposer, PV, width, height,
            invPV*LinAlg::Fill(1.0f, -cameraAspectRatio, far, 1.0f), invPV*LinAlg::Fill(1.0f, cameraAspectRatio, far, 1.0f));

        projectLineToSvg(svgComposer, PV, width, height,
            invPV*LinAlg::Fill(1.0f, cameraAspectRatio, far, 1.0f), invPV*LinAlg::Fill(-1.0f, cameraAspectRatio, far, 1.0f));

        projectLineToSvg(svgComposer, PV, width, height,
            invPV*LinAlg::Fill(-1.0f, cameraAspectRatio, far, 1.0f), invPV*LinAlg::Fill(-1.0f, -cameraAspectRatio, far, 1.0f));



        projectLineToSvg(svgComposer, PV, width, height,
            invPV*LinAlg::Fill(-1.0f, -cameraAspectRatio, near, 1.0f), invPV*LinAlg::Fill(-1.0f, -cameraAspectRatio, far, 1.0f));

        projectLineToSvg(svgComposer, PV, width, height,
            invPV*LinAlg::Fill(1.0f, -cameraAspectRatio, near, 1.0f), invPV*LinAlg::Fill(1.0f, -cameraAspectRatio, far, 1.0f));

        projectLineToSvg(svgComposer, PV, width, height,
            invPV*LinAlg::Fill(1.0f, cameraAspectRatio, near, 1.0f), invPV*LinAlg::Fill(1.0f, cameraAspectRatio, far, 1.0f));

        projectLineToSvg(svgComposer, PV, width, height,
            invPV*LinAlg::Fill(-1.0f, cameraAspectRatio, near, 1.0f), invPV*LinAlg::Fill(-1.0f, cameraAspectRatio, far, 1.0f));

    }
}


int main(int argc, char **argv)
{
#if 1
    if (argc != 2) {
        std::cout << "A filename to the .binsfm file must be supplied!" << std::endl;
        return 0;
    }
#endif

    Engine::Platform::X11Window window;

    Engine::Platform::BaseWindow::WindowParams windowParams;

    windowParams.width = 1280;
    windowParams.height = 720;
    windowParams.windowName = "Sparse reconstruction viewer";
    windowParams.visualAttributes.push_back(GLX_X_RENDERABLE);
    windowParams.visualAttributes.push_back(True);
    windowParams.visualAttributes.push_back(GLX_DRAWABLE_TYPE);
    windowParams.visualAttributes.push_back(GLX_WINDOW_BIT);
    windowParams.visualAttributes.push_back(GLX_X_VISUAL_TYPE);
    windowParams.visualAttributes.push_back(GLX_TRUE_COLOR);
    windowParams.visualAttributes.push_back(GLX_RED_SIZE);
    windowParams.visualAttributes.push_back(8);
    windowParams.visualAttributes.push_back(GLX_GREEN_SIZE);
    windowParams.visualAttributes.push_back(8);
    windowParams.visualAttributes.push_back(GLX_BLUE_SIZE);
    windowParams.visualAttributes.push_back(8);
    windowParams.visualAttributes.push_back(GLX_ALPHA_SIZE);
    windowParams.visualAttributes.push_back(8);

    windowParams.visualAttributes.push_back(GLX_DEPTH_SIZE);
    windowParams.visualAttributes.push_back(24);
    windowParams.visualAttributes.push_back(GLX_STENCIL_SIZE);
    windowParams.visualAttributes.push_back(8);

    windowParams.visualAttributes.push_back(GLX_DOUBLEBUFFER);
    windowParams.visualAttributes.push_back(True);
    windowParams.visualAttributes.push_back(None);

    windowParams.contextAttributes.push_back(GLX_CONTEXT_MAJOR_VERSION_ARB);
    windowParams.contextAttributes.push_back(1);
    windowParams.contextAttributes.push_back(GLX_CONTEXT_MINOR_VERSION_ARB);
    windowParams.contextAttributes.push_back(2);
#ifdef DEBUG_CONTEXT
    windowParams.contextAttributes.push_back(GLX_CONTEXT_FLAGS_ARB);
    windowParams.contextAttributes.push_back(GLX_CONTEXT_DEBUG_BIT_ARB);
#endif

/*
    windowParams.contextAttributes.push_back(GLX_CONTEXT_FLAGS_ARB);
    windowParams.contextAttributes.push_back(GLX_CONTEXT_FORWARD_COMPATIBLE_BIT_ARB);
*/
/*
    windowParams.contextAttributes.push_back(GLX_CONTEXT_PROFILE_MASK_ARB);
    windowParams.contextAttributes.push_back(GLX_CONTEXT_CORE_PROFILE_BIT_ARB);
*/
    windowParams.contextAttributes.push_back(None);

    windowParams.fullscreen = false;
    windowParams.grabInput = false;

    window.resetWindow(windowParams);

    Engine::Graphics::RenderDevice renderDevice;
    renderDevice.switchRenderContext(window.getContext());

    SparseReconstruction sparseReconstruction;
    if (argc > 1)
        sparseReconstruction.load(argv[1]);
    else {
        //sparseReconstruction.load("bark1.binSFM");
        //sparseReconstruction.load("fountain-P11.binSFM");
        //sparseReconstruction.load("Herz-Jesu-P25.binSFM");
        //sparseReconstruction.load("castle-P30.binSFM");
        //sparseReconstruction.load("Snail.binSFM");
        //sparseReconstruction.load("rock1Undist.binSFM");
        //sparseReconstruction.load("sparseReconstruction.binSFM");
        //sparseReconstruction.load("../CommandLineFrontend/HerzJesuP25.binSFM");
        //sparseReconstruction.load("../CommandLineFrontend/kaktus.binSFM");
        //sparseReconstruction.load("VolcanicBomb.binSFM");
        //sparseReconstruction.load("volcano.binSFM");
        //sparseReconstruction.load("VolcanicBomb_PARTIAL.binSFM");
        //sparseReconstruction.load("Turntable.binSFM");

       // sparseReconstruction.storeAsPlyPointCloud("vulcano.ply");
    }

    std::cout << "Running window on " << glGetString(GL_VENDOR) << std::endl;
    glewInit();


    LinAlg::Vector3f cameraAccel;
    SpectatorCamera camera;

    LinAlg::Matrix4x4f projectionMatrix;
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(30.0f, windowParams.width / (float)windowParams.height, 0.01f, 100.0f);
    glGetFloatv(GL_PROJECTION_MATRIX, &projectionMatrix[0][0]);
    projectionMatrix = projectionMatrix.T();
    glMatrixMode(GL_MODELVIEW);

    glEnable(GL_DEPTH_TEST);
    Engine::Graphics::BufferObject pointBuffer(&renderDevice, Engine::Graphics::BufferObject::BUFFER_TYPE_ARRAY);


    const float deltaT = 1.0f/60.0f;

    bool operate = false;

    float camSize = 100.0f;

    //const float aspectRatio = 2.0f/3.0f;
    const float aspectRatio = 3.0f/4.0f;
    //const float aspectRatio = 2848.0f/4272.0f;


    bool shutdown = false;
    while (!shutdown) {

        camera.accelerate(cameraAccel, deltaT);
        camera.operate(deltaT);
        camera.updateMatrices();

        {
            glMatrixMode(GL_MODELVIEW);
            LinAlg::Matrix4x4f M = camera.getWorldToEyeSpaceMatrix().T();
            glLoadMatrixf(&M[0][0]);
        }

        glViewport(0, 0, 1280, 720);
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);


        glBegin(GL_LINES);
        glColor3f(1.0f, 0.0f, 0.0f);
        glVertex3f(0.0f, 0.0f, 0.0f);
        glVertex3f(1.0f, 0.0f, 0.0f);

        glColor3f(0.0f, 1.0f, 0.0f);
        glVertex3f(0.0f, 0.0f, 0.0f);
        glVertex3f(0.0f, 1.0f, 0.0f);

        glColor3f(0.0f, 0.0f, 1.0f);
        glVertex3f(0.0f, 0.0f, 0.0f);
        glVertex3f(0.0f, 0.0f, 1.0f);
        glEnd();

        glColor3f(1.0f, 1.0f, 1.0f);
        for (unsigned i = 0; i < sparseReconstruction.getCameras().size(); i++) {
            const LinAlg::Matrix4x4f &InvPV = sparseReconstruction.getCameras()[i].invPV;
            const float near = camSize;
            const float far = camSize / 10.0f;
            glBegin(GL_LINE_LOOP);
            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(-1.0f, -aspectRatio, near, 1.0f);
                glVertex4fv(&V[0]);
            }
            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(1.0f, -aspectRatio, near, 1.0f);
                glVertex4fv(&V[0]);
            }
            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(1.0f, aspectRatio, near, 1.0f);
                glVertex4fv(&V[0]);
            }
            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(-1.0f, aspectRatio, near, 1.0f);
                glVertex4fv(&V[0]);
            }
            glEnd();
            glBegin(GL_LINE_LOOP);
            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(-1.0f, -aspectRatio, far, 1.0f);
                glVertex4fv(&V[0]);
            }
            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(1.0f, -aspectRatio, far, 1.0f);
                glVertex4fv(&V[0]);
            }
            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(1.0f, aspectRatio, far, 1.0f);
                glVertex4fv(&V[0]);
            }
            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(-1.0f, aspectRatio, far, 1.0f);
                glVertex4fv(&V[0]);
            }
            glEnd();
            glBegin(GL_LINES);
            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(-1.0f, -aspectRatio, near, 1.0f);
                glVertex4fv(&V[0]);
            }
            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(-1.0f, -aspectRatio, far, 1.0f);
                glVertex4fv(&V[0]);
            }

            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(1.0f, -aspectRatio, near, 1.0f);
                glVertex4fv(&V[0]);
            }
            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(1.0f, -aspectRatio, far, 1.0f);
                glVertex4fv(&V[0]);
            }

            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(1.0f, aspectRatio, near, 1.0f);
                glVertex4fv(&V[0]);
            }
            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(1.0f, aspectRatio, far, 1.0f);
                glVertex4fv(&V[0]);
            }

            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(-1.0f, aspectRatio, near, 1.0f);
                glVertex4fv(&V[0]);
            }
            {
                LinAlg::Vector4f V = InvPV * LinAlg::Fill(-1.0f, aspectRatio, far, 1.0f);
                glVertex4fv(&V[0]);
            }
            glEnd();
        }

        glPointSize(8.0f);
        #if 1
        glBegin(GL_POINTS);
        for (unsigned i = 0; i < sparseReconstruction.getTracks().size(); i++) {
            LinAlg::Vector3f c = LinAlg::Fill<float>((sparseReconstruction.getTracks()[i].color >> 0) & 0xFF,
                                                     (sparseReconstruction.getTracks()[i].color >> 8) & 0xFF,
                                                     (sparseReconstruction.getTracks()[i].color >> 16) & 0xFF) / 255.0f;
            glColor3fv(&c[0]);
            glVertex4fv(&sparseReconstruction.getTracks()[i].position[0]);
        }
        glEnd();
        #else
        renderDevice.bindBuffer(&pointBuffer);

        glVertexPointer(3, GL_FLOAT, 6*4, NULL);
        glColorPointer(3, GL_FLOAT, 6*4, (void*)(3*4));

        glEnableClientState(GL_VERTEX_ARRAY);
        glEnableClientState(GL_COLOR_ARRAY);

        glDrawArrays(GL_POINTS, 0, pointsInBuffer);

        glDisableClientState(GL_VERTEX_ARRAY);
        glDisableClientState(GL_COLOR_ARRAY);
        glBindBuffer(GL_ARRAY_BUFFER, 0);
        #endif

        Engine::Platform::BaseWindow::Event event;
        while (window.getNextEvent(event)) {
            // check for messages
            switch (event.type) {
                case Engine::Platform::BaseWindow::Event::TYPE_KEY_DOWN:
                    switch (event.key.keysym) {
                        case 'w':
                            cameraAccel[2] = (window.getState().buttonState & (1 << 3))?-10.0f:-1.0f;
                        break;
                        case 's':
                            cameraAccel[2] = (window.getState().buttonState & (1 << 3))?10.0f:1.0f;
                        break;
                        case 'a':
                            cameraAccel[0] = (window.getState().buttonState & (1 << 3))?-10.0f:-1.0f;
                        break;
                        case 'd':
                            cameraAccel[0] = (window.getState().buttonState & (1 << 3))?10.0f:1.0f;
                        break;
                        case 'c':
                            cameraAccel[1] = (window.getState().buttonState & (1 << 3))?-10.0f:-1.0f;
                        break;
                        case ' ':
                            cameraAccel[1] = (window.getState().buttonState & (1 << 3))?10.0f:1.0f;
                        break;
                        case 'r':
                            //sparseReconstruction.rasterPointImage("test.png", 1280, 720, 4.0f, projectionMatrix * camera.getWorldToEyeSpaceMatrix());
                            sparseReconstruction.drawSVGImage("test.svg", 1920, 1080, 8.0f, projectionMatrix * camera.getWorldToEyeSpaceMatrix(), camSize, aspectRatio);
                        break;
                        case '+':
                            camSize *= 0.9f;
                        break;
                        case '-':
                            camSize /= 0.9f;
                        break;
                        default:
                        break;
                    }
                break;
                case Engine::Platform::BaseWindow::Event::TYPE_KEY_UP:
                    switch (event.key.keysym) {
                        case 'w':
                        case 's':
                            cameraAccel[2] = 0.0f;
                        break;
                        case 'a':
                        case 'd':
                            cameraAccel[0] = 0.0f;
                        break;
                        case 'c':
                        case ' ':
                            cameraAccel[1] = 0.0f;
                        break;
                        default:
                        break;
                    }
                break;
                case Engine::Platform::BaseWindow::Event::TYPE_MOTION:
                    if (window.getState().buttonState & (1 << 1)) {
                        camera.turn(-event.motion.relY * 0.01f, event.motion.relX * 0.01f);
                    }
                break;
                case Engine::Platform::BaseWindow::Event::TYPE_BUTTON_DOWN:
                break;
                case Engine::Platform::BaseWindow::Event::TYPE_NOTIFICATION:
                    if (event.notification.message == Engine::Platform::BaseWindow::Event::Notification::NM_WMSHUTDOWN)
                        shutdown = true;
                break;
                default:
                break;
            }
        }
        glFlush();
        window.getContext()->swap();

    }
    return 0;
}
