/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "Mesh.h"

#include "RenderDevice.h"

#include <sstream>
#include <stdexcept>
#include <boost/lexical_cast.hpp>
#include "BufferObject.h"

namespace Engine {
namespace Graphics {





Mesh::Mesh(RenderDevice *renderDevice) : m_renderDevice(renderDevice)
{
    m_indexBuffer = nullptr;
}

Mesh::~Mesh()
{
    for (unsigned i = 0; i < m_attachedAttribBuffers.size(); i++)
        delete m_attachedAttribBuffers[i].m_vertexBuffer;

    delete m_indexBuffer;
}

void Mesh::setupVertexBuffer(unsigned numBuffer, MeshBufferLayout *layouts)
{
    for (unsigned i = 0; i < m_attachedAttribBuffers.size(); i++) {
        delete m_attachedAttribBuffers[i].m_vertexBuffer;
    }

    m_attachedAttribBuffers.resize(numBuffer);
    for (unsigned i = 0; i < m_attachedAttribBuffers.size(); i++) {
        m_attachedAttribBuffers[i].m_vertexBuffer = new BufferObject(m_renderDevice, BufferObject::BUFFER_TYPE_ARRAY);
        m_attachedAttribBuffers[i].m_layout = layouts[i];
    }
}

void Mesh::fillVertexBuffer(unsigned bufferIndex, unsigned size, const void *data)
{
    m_attachedAttribBuffers[bufferIndex].m_vertexBuffer->resize(size, BufferObject::USAGE_TYPE_STATIC_DRAW);
    m_attachedAttribBuffers[bufferIndex].m_vertexBuffer->fillBuffer(0, size, data);
}

void Mesh::setupIndexData(unsigned numRanges, MeshIndexRange *ranges, unsigned bufferDataType, unsigned bufferSize, const void *bufferData)
{
    m_indexRanges.resize(numRanges);
    for (unsigned i = 0; i < m_indexRanges.size(); i++)
        m_indexRanges[i] = ranges[i];

    m_indexDataType = bufferDataType;
    if (m_indexBuffer == NULL)
        m_indexBuffer = new BufferObject(m_renderDevice, BufferObject::BUFFER_TYPE_ELEMENT_ARRAY);

    m_indexBuffer->resize(bufferSize, BufferObject::USAGE_TYPE_STATIC_DRAW);
    m_indexBuffer->fillBuffer(0, bufferSize, bufferData);
}

unsigned Mesh::getAttribBufferIndex(const std::string &name) const
{
    for (unsigned i = 0; i < m_attachedAttribBuffers.size(); i++)
        if (m_attachedAttribBuffers[i].m_layout.name == name)
            return i;

    return -1;
}

unsigned Mesh::getIndexRangeIndex(const std::string &name) const
{
    for (unsigned i = 0; i < m_indexRanges.size(); i++)
        if (m_indexRanges[i].name == name)
            return i;

    return -1;
}



}
}
