/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "RenderDevice.h"
#include "Texture1D.h"
#include "Texture2D.h"
#include "Texture2DArray.h"
#include "FrameBufferObject.h"
#include "BufferObject.h"
#include "Mesh.h"
#include "VertexArrayObject.h"

#include "GLSLProgram.h"

#include "../platform/BaseWindow.h"

namespace Engine {
namespace Graphics {



RenderDevice::RenderDevice()
{
    //ctor
}

RenderDevice::~RenderDevice()
{
    //dtor
}

void RenderDevice::switchRenderContext(Platform::RenderContextInterface *renderContext)
{
    m_renderContext = renderContext;
    m_renderContext->makeCurrent();

    m_renderDeviceState.boundVAO = -1;
    for (unsigned i = 0; i < 8; i++)
        m_renderDeviceState.boundTextures[i] = -1;

    m_renderDeviceState.boundFBO = -1;
    m_renderDeviceState.boundProgram = -1;
}


void RenderDevice::bindTexture(Texture2D *texture)
{
    bindTexture(texture, 7);
}

void RenderDevice::bindTexture(Texture2D *texture, unsigned samplerIndex)
{
    if (m_renderDeviceState.boundTextures[samplerIndex] != texture->getID()) {
        glActiveTexture(GL_TEXTURE0+samplerIndex);
        glBindTexture(GL_TEXTURE_2D, texture->m_textureID);
        m_renderDeviceState.boundTextures[samplerIndex] = texture->getID();
    }
}

void RenderDevice::bindTexture2DArray(Texture2DArray *texture)
{
    bindTexture2DArray(texture, 7);
}

void RenderDevice::bindTexture2DArray(Texture2DArray *texture, unsigned samplerIndex)
{
    if (m_renderDeviceState.boundTextures[samplerIndex] != texture->getID()) {
        glActiveTexture(GL_TEXTURE0+samplerIndex);
        glBindTexture(GL_TEXTURE_2D_ARRAY, texture->m_textureID);
        m_renderDeviceState.boundTextures[samplerIndex] = texture->getID();
    }
}


void RenderDevice::bindTexture1D(Texture1D *texture)
{
    bindTexture1D(texture, 7);
}

void RenderDevice::bindTexture1D(Texture1D *texture, unsigned samplerIndex)
{
    if (m_renderDeviceState.boundTextures[samplerIndex] != texture->getID()) {
        glActiveTexture(GL_TEXTURE0+samplerIndex);
        glBindTexture(GL_TEXTURE_1D, texture->m_textureID);
        m_renderDeviceState.boundTextures[samplerIndex] = texture->getID();
    }
}



void RenderDevice::bindFramebufferObject(FrameBufferObject *fbo)
{
    if (m_renderDeviceState.boundFBO != fbo->m_objectID) {
        glBindFramebuffer(GL_FRAMEBUFFER, fbo->m_objectID);
        m_renderDeviceState.boundFBO = fbo->m_objectID;
    }
}

void RenderDevice::bindDefaultFramebufferObject()
{
    if (m_renderDeviceState.boundFBO != 0) {
        glBindFramebuffer(GL_FRAMEBUFFER, 0);
        m_renderDeviceState.boundFBO = 0;
    }
}


void RenderDevice::setDrawBuffers(unsigned count, const unsigned *attachments)
{
    glDrawBuffers(count, attachments);
}

void RenderDevice::bindBuffer(BufferObject *buffer)
{
    if (m_renderDeviceState.boundVAO != (unsigned)-1) {
        glBindVertexArray(0);
        m_renderDeviceState.boundVAO = -1;
    }
    glBindBuffer(buffer->m_bufferType, buffer->m_bufferId);
}

void RenderDevice::bindProgram(GLSLProgram *program)
{
    if (program->m_programID != m_renderDeviceState.boundProgram) {
        glUseProgram(program->m_programID);
        m_renderDeviceState.boundProgram = program->m_programID;
    }
}

void RenderDevice::bindUniformBufferToSlot(BufferObject *buffer, unsigned slot, unsigned offset, unsigned size)
{
    if ((offset == 0) && (size == (unsigned)-1))
        glBindBufferBase(GL_UNIFORM_BUFFER, slot, buffer->m_bufferId);
    else
        glBindBufferRange(GL_UNIFORM_BUFFER, slot, buffer->m_bufferId, offset, size);
}



void RenderDevice::bindVertexArrayObject(const VertexArrayObject *vao)
{
    if (m_renderDeviceState.boundVAO != vao->m_objectId) {
        glBindVertexArray(vao->m_objectId);
        m_renderDeviceState.boundVAO = vao->m_objectId;
    }
}


void RenderDevice::renderIndices(const VertexArrayObject *vao, PrimitiveType primitiveType, unsigned indexDataType, unsigned offset, unsigned count)
{
    bindVertexArrayObject(vao);
    if (indexDataType == GL_UNSIGNED_SHORT)
        glDrawElements(primitiveType, count, indexDataType, (void*)(size_t) (offset*2));
    else
        if (indexDataType == GL_UNSIGNED_INT)
            glDrawElements(primitiveType, count, indexDataType, (void*)(size_t) (offset*4));
}

void RenderDevice::renderVertices(const VertexArrayObject *vao, PrimitiveType primitiveType, unsigned offset, unsigned count)
{
    bindVertexArrayObject(vao);
    glDrawArrays(primitiveType, offset, count);
}
/*
void RenderDevice::executeRenderCommandBuffer(const RenderCommandBuffer &commandBuffer, unsigned pass)
{
    for (unsigned i = 0; i < commandBuffer.m_passes[pass].m_commands.size(); i++)
        commandBuffer.m_passes[pass].m_commands[i](this);
}

*/

}
}
