/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef PATCHDESCRIPTORDB_H
#define PATCHDESCRIPTORDB_H

#include <stdint.h>
#include <vector>
#include "../tools/LinAlg.h"

namespace SFM {
// no longer needed
class PatchDescriptorDB
{
    public:
        PatchDescriptorDB(unsigned gridDim, unsigned gridRes);
        ~PatchDescriptorDB();

        struct PatchDescriptor {
            unsigned userData;
            float score;
            unsigned projectionIndex;
            uint32_t data[16*16] __attribute__((aligned (16)));

            float computeSquaredDifference(const PatchDescriptor &other) const;
        };

        void computeProjectionIndices(PatchDescriptor *descriptors, unsigned count) const;
        void compile(PatchDescriptor *descriptors, unsigned count);

        struct Match {
            unsigned srcIndex;
            unsigned closestMatchIndex;
            float sqrDiff;
            float normalizedDiff;
            inline bool operator<(const Match &other) const { return normalizedDiff < other.normalizedDiff; }
        };


        struct MatchConstraint {
            virtual unsigned getNumDescriptors() const = 0;
            virtual const PatchDescriptor **getDescriptors() = 0;
            virtual bool preMatchTest(unsigned sourceIndex, unsigned destinationIndex) = 0;
            virtual void postMatchOperation(unsigned sourceIndex, unsigned destinationIndex, float normalizedMatchScore) = 0;
        };

        void findMatchesConstrained(MatchConstraint *constraint) const;

        void findMatches(const PatchDescriptor *descriptors, unsigned count, std::vector<Match> &matches) const;
        void findMatches(const PatchDescriptorDB &other, std::vector<Match> &matches) const;

        inline unsigned getNumPatchDescriptors() const { return m_descriptors.size(); }
        const PatchDescriptor &getPatchDescriptor(unsigned index) const { return m_descriptors[index]; }


        void recomputeScores(const std::vector<LinAlg::Vector2f> &screenPositions);

    protected:
        void computeProjectionIndicesSubrange(PatchDescriptor *descriptors, unsigned count) const;
        void findMatchesSubrange(const PatchDescriptor *descriptors, unsigned count, unsigned offset, Match *matches) const;
        void findMatchesConstrainedSubrange(MatchConstraint *constraint, const PatchDescriptor **descriptors, unsigned offset, unsigned count) const;
        void recomputeScoresSubrange(const std::vector<LinAlg::Vector2f> &screenPositions, unsigned offset, unsigned count);


        struct GridEntry {
            unsigned start;
            unsigned count;
        };

        struct GridBasis
        {
            LinAlg::Vector<16*16*3, float> offset[8];
            LinAlg::Vector<16*16*3, float> direction[8];
        };

        GridBasis m_basis;
        float m_projectionBasis[16*16*3*8] __attribute__((aligned (16)));
        float m_projectionOffset[8] __attribute__((aligned (16)));

        unsigned m_gridDim;
        unsigned m_gridResolution;

        std::vector<PatchDescriptor> m_descriptors;
        std::vector<GridEntry> m_grid;


};

}

#endif // PATCHDESCRIPTORDB_H
