/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "SiftDescriptorDB.h"


#include <xmmintrin.h>
#include <immintrin.h>

#include <stdexcept>

extern const unsigned char SiftGridDBBasisData[];
extern const unsigned SiftGridDBBasisData_size;

namespace SFM {


SiftDescriptorDB::SiftDescriptorDB(unsigned gridDim, unsigned gridRes)
{
#if 0    
    FILE *file = fopen("SiftGridDBBasis.bin", "rb");
    if (file == nullptr)
        throw std::runtime_error("Could not open file 'SiftGridDBBasis.bin'");
    if (fread(&m_basis, 1, sizeof(m_basis), file) != sizeof(m_basis))
        throw std::runtime_error("Could not read file 'SiftGridDBBasis.bin'");

    fclose(file);
#else
    if (sizeof(m_basis) != SiftGridDBBasisData_size)
        throw std::runtime_error("Internal error: The stored sift grid DB basis is broken");
    memcpy(&m_basis, SiftGridDBBasisData, sizeof(m_basis));
#endif


    m_gridDim = gridDim;
    m_gridResolution = gridRes;

    for (unsigned i = 0; i < 8; i++) {
        float absMax = fabs(m_basis.direction[i][0]);
        for (unsigned y = 0; y < 128; y++)
            absMax = fmax(absMax, fabs(m_basis.direction[i][y]));

        m_projectionBasisScales[i] = absMax / 32700.0f;
        float f = 32700.0f / absMax;

        for (unsigned y = 0; y < 128; y++)
            m_projectionBasis[i*128+y] = m_basis.direction[i][y] * f;

        m_projectionOffsets[i] = 0.5f;
        for (unsigned y = 0; y < 128; y++)
            m_projectionOffsets[i] += m_basis.offset[i][y];
    }



    unsigned numLookupCells = 1;
    for (unsigned i = 0; i < m_gridDim; i++)
        numLookupCells *= 3;

    m_lookupOffsets.reserve(numLookupCells);

    int counters[m_gridDim];
    memset(counters, 0, m_gridDim*sizeof(int));

    for (unsigned i = 0; i < numLookupCells; i++) {
        int offset = 0;
        for (unsigned j = 0; j < m_gridDim; j++) {
            offset = offset * (int)m_gridResolution + (counters[m_gridDim-1-j]-1);
        }
        m_lookupOffsets.push_back(offset);

        for (unsigned j = 0; j < m_gridDim; j++) {
            counters[j]++;
            if (counters[j] == 3)
                counters[j] = 0;
            else break;
        }
    }
    std::sort(m_lookupOffsets.begin(), m_lookupOffsets.end());
}

SiftDescriptorDB::~SiftDescriptorDB()
{
    //dtor
}


void SiftDescriptorDB::computeProjectionIndicesSubrange(SiftDescriptor *descriptors, unsigned count) const
{
    /*
    __m128i bias = _mm_setr_epi16(0, 0, -128, -128,
                                  0, 0, -128, -128);
    */

    for (unsigned patchIndex = 0; patchIndex < count; patchIndex++) {
        __m128i sums[8];
        for (unsigned i = 0; i < 8; i++)
            sums[i] = _mm_setzero_si128();

        __m128i sqrLength = _mm_setzero_si128();

        for (unsigned i = 0; i < 128/16; i++) {
            {
                __m128i srcABCD = _mm_load_si128((const __m128i*)(descriptors[patchIndex].descriptor + i*4));

                __m128i firstHalf = _mm_unpacklo_epi8(srcABCD, _mm_setzero_si128());
                __m128i secondHalf = _mm_unpackhi_epi8(srcABCD, _mm_setzero_si128());
/*
                firstHalf = _mm_add_epi16(firstHalf, bias);
                secondHalf = _mm_add_epi16(secondHalf, bias);
*/
                sqrLength = _mm_add_epi32(sqrLength, _mm_madd_epi16(firstHalf, firstHalf));
                sqrLength = _mm_add_epi32(sqrLength, _mm_madd_epi16(secondHalf, secondHalf));


                for (unsigned j = 0; j < 8; j++) {
                    sums[j] = _mm_add_epi32(sums[j], _mm_madd_epi16(firstHalf, _mm_load_si128((const __m128i*)(m_projectionBasis + i*16 + 0 + j * 128))));
                    sums[j] = _mm_add_epi32(sums[j], _mm_madd_epi16(secondHalf, _mm_load_si128((const __m128i*)(m_projectionBasis + i*16 + 8 + j * 128))));
                }
            }
        }

        unsigned totalSqrLength = _mm_extract_epi32(sqrLength, 0) +
                                  _mm_extract_epi32(sqrLength, 1) +
                                  _mm_extract_epi32(sqrLength, 2) +
                                  _mm_extract_epi32(sqrLength, 3);

        descriptors[patchIndex].rcpLength = 1.0f / sqrtf(totalSqrLength);

        unsigned buckets[8];
        for (unsigned i = 0; i < 8; i++) {
            float v = (_mm_extract_epi32(sums[i], 0) +
                       _mm_extract_epi32(sums[i], 1) +
                       _mm_extract_epi32(sums[i], 2) +
                       _mm_extract_epi32(sums[i], 3)) * descriptors[patchIndex].rcpLength * m_projectionBasisScales[i] + m_projectionOffsets[i];

            v *= m_gridResolution;
            buckets[i] = std::min<int>(std::max<int>(v, 0), m_gridResolution-1);
        }
/*
        std::cout << "Coordinate for " << patchIndex <<": ";
        for (unsigned i = 0; i < 5; i++)
            std::cout << buckets[i] << " ";
        std::cout << std::endl;
*/
        unsigned projectionIndex = buckets[7];
        for (int i = 6; i >= 0; i--)
            projectionIndex = projectionIndex * m_gridResolution + buckets[i];

        descriptors[patchIndex].projectionIndex = projectionIndex;
    }
}

void SiftDescriptorDB::computeProjectionIndices(SiftDescriptor *descriptors, unsigned count) const
{
#if 1
    if (count > 100) {
        TaskGroup group;
        for (unsigned i = 0; i < count; i+= 100) {
            group.add(
                      boost::bind(&SiftDescriptorDB::computeProjectionIndicesSubrange, this, descriptors + i,
                                  std::min<unsigned>(100, count-i)),
                      TaskScheduler::get());
        }
        TaskScheduler::get().waitFor(&group);
    } else
#endif
        computeProjectionIndicesSubrange(descriptors, count);

}



struct PatchGridEntry {
    unsigned patchIndex;
    unsigned gridIndex;
    inline bool operator<(const PatchGridEntry &other) const { return gridIndex < other.gridIndex; }
};


void SiftDescriptorDB::compile(SiftDescriptor *descriptors, unsigned count)
{
    m_descriptors.resize(count);

    memcpy(&m_descriptors[0], descriptors, count*sizeof(SiftDescriptor));
    computeProjectionIndices(&m_descriptors[0], count);



    unsigned gridCellCount = 1;
    for (unsigned i = 0; i < m_gridDim; i++)
        gridCellCount *= m_gridResolution;

    std::vector<PatchGridEntry> entries;
    entries.resize(m_descriptors.size());
    for (unsigned i = 0; i < m_descriptors.size(); i++) {
        entries[i].patchIndex = i;
        entries[i].gridIndex = m_descriptors[i].projectionIndex % gridCellCount;
    }
    std::sort(entries.begin(), entries.end());


    m_grid.resize(gridCellCount);
    memset(&m_grid[0], 0, m_grid.size()*sizeof(GridEntry));
    std::vector<SiftDescriptor> sortedDescriptors;
    sortedDescriptors.resize(m_descriptors.size());
    for (unsigned i = 0; i < entries.size(); i++) {
        sortedDescriptors[i] = m_descriptors[entries[i].patchIndex];
        if (i > 0) {
            if (entries[i-1].gridIndex != entries[i].gridIndex) {
                m_grid[entries[i-1].gridIndex].count = i-m_grid[entries[i-1].gridIndex].start;
                m_grid[entries[i].gridIndex].start = i;
            }
        }
    }
    if (entries.size() > 0)
        m_grid[entries[entries.size()-1].gridIndex].count = entries.size() - m_grid[entries[entries.size()-1].gridIndex].start;

    sortedDescriptors.swap(m_descriptors);
}

template<unsigned index>
inline float extract_ps(const __m128 &v)
{
    return _mm_cvtss_f32(_mm_shuffle_ps(v, v, _MM_SHUFFLE(0, 0, 0, index)));
}


static const __m128i singleByteMask = _mm_set1_epi32(0xFF);
static const __m128 factor = _mm_set1_ps(1.0f/(255.0f*255.0f));

float SiftDescriptorDB::SiftDescriptor::computeDifference(const SiftDescriptor &other) const
{
#if 1
    __m128i sums = _mm_setzero_si128();
/*
    __m128i bias = _mm_setr_epi16(0, 0, -128, -128,
                                  0, 0, -128, -128);
*/
    for (unsigned i = 0; i < 128/16; i++) {
        __m128i src1 = _mm_load_si128((const __m128i*)(other.descriptor + i*4));
        __m128i src2 = _mm_load_si128((const __m128i*)(descriptor + i*4));

        __m128i firstHalf1 = _mm_unpacklo_epi8(src1, _mm_setzero_si128());
        __m128i secondHalf1 = _mm_unpackhi_epi8(src1, _mm_setzero_si128());
        __m128i firstHalf2 = _mm_unpacklo_epi8(src2, _mm_setzero_si128());
        __m128i secondHalf2 = _mm_unpackhi_epi8(src2, _mm_setzero_si128());
/*
        firstHalf1 = _mm_add_epi16(firstHalf1, bias);
        secondHalf1 = _mm_add_epi16(secondHalf1, bias);
        firstHalf2 = _mm_add_epi16(firstHalf2, bias);
        secondHalf2 = _mm_add_epi16(secondHalf2, bias);
*/
        sums = _mm_add_epi32(sums, _mm_madd_epi16(firstHalf1, firstHalf2));
        sums = _mm_add_epi32(sums, _mm_madd_epi16(secondHalf1, secondHalf2));
    }

    unsigned totalSum = _mm_extract_epi32(sums, 0) +
                        _mm_extract_epi32(sums, 1) +
                        _mm_extract_epi32(sums, 2) +
                        _mm_extract_epi32(sums, 3);
    return 1.0f - totalSum * rcpLength * other.rcpLength;
#elif 0
    unsigned char *ptr1 = (unsigned char*) data;
    unsigned char *ptr2 = (unsigned char*) other.data;
    unsigned sum = 0;
    unsigned length1 = 0;
    unsigned length2 = 0;
    for (unsigned i = 0; i < 128; i++) {
        length1 += (uint16_t)ptr1[i]*(uint16_t)ptr1[i];
        length2 += (uint16_t)ptr2[i]*(uint16_t)ptr2[i];
        sum += (uint16_t)ptr1[i] * (uint16_t)ptr2[i];
    }

    return 1.0f - sum / sqrtf((uint64_t)length1 * (uint64_t)length2);
#else
    unsigned char *ptr1 = (unsigned char*) descriptor;
    unsigned char *ptr2 = (unsigned char*) other.descriptor;
    uint64_t sum = 0;
    uint64_t length1 = 0;
    uint64_t length2 = 0;
    for (unsigned i = 0; i < 128; i++) {
        length1 += (uint32_t)ptr1[i]*(uint32_t)ptr1[i];
        length2 += (uint32_t)ptr2[i]*(uint32_t)ptr2[i];
        sum += (uint32_t)ptr1[i] * (uint32_t)ptr2[i];
    }

    return 1.0f - sum / sqrtf((uint64_t)length1 * (uint64_t)length2);
#endif
}

void SiftDescriptorDB::findMatchesSubrange(const SiftDescriptor *descriptors, unsigned count, unsigned offset, Match *matches) const
{
    #if 1
    /*
    unsigned numGridCellsToVisit = 1;
    for (unsigned i = 0; i < m_gridDim; i++)
        numGridCellsToVisit *= 3;
*/
    //unsigned totalNumCompares = 0;

    for (unsigned patchIndex = 0; patchIndex < count; patchIndex++) {
        unsigned bestMatch = -1;
        float bestMatchSqrDiff = 1e30f;
        float bestMatchNormDiff = 1e30f;
        //float secondBestMatch

        const SiftDescriptor &otherPatch = descriptors[patchIndex];
        const unsigned otherPatchCell = otherPatch.projectionIndex % m_grid.size();
/*
        for (unsigned cell = 0; cell < numGridCellsToVisit; cell++) {

            bool validCell = true;

            unsigned cellIndex;
            {
                unsigned bucketsIndices[8];

                unsigned srcCellIndex = otherPatch.projectionIndex % m_grid.size();
                unsigned destrCell = cell;
                for (unsigned i = 0; i < m_gridDim; i++) {
                    int bucket = srcCellIndex % m_gridResolution;
                    int bucketOffset = (int)(destrCell % 3) - 1;

                    bucket += bucketOffset;

                    if ((bucket < 0) || (bucket >= (int)m_gridResolution)) {
                        validCell = false;
                        break;
                    }

                    bucketsIndices[i] = bucket;

                    srcCellIndex /= m_gridResolution;
                    destrCell /= 3;
                }

                if (!validCell)
                    continue;
                cellIndex = 0;
                for (unsigned i = 0; i < m_gridDim; i++) {
                    cellIndex = cellIndex * m_gridResolution + bucketsIndices[m_gridDim-1-i];
                }
            }
            */
        for (unsigned cell = 0; cell < m_lookupOffsets.size(); cell++) {
            unsigned cellIndex = otherPatchCell + m_lookupOffsets[cell];
            if (cellIndex >= m_grid.size())
                continue;

            for (unsigned selfPatchIndex = m_grid[cellIndex].start; selfPatchIndex < m_grid[cellIndex].start+m_grid[cellIndex].count; selfPatchIndex++) {
            //for (unsigned selfPatchIndex = 0; selfPatchIndex < m_candidates.size(); selfPatchIndex++) {

                const SiftDescriptor &thisPatch = m_descriptors[selfPatchIndex];


                float sum = thisPatch.computeDifference(otherPatch);
                //totalNumCompares++;

                if (sum < bestMatchSqrDiff) {
                    bestMatchSqrDiff = sum;
                    bestMatchNormDiff = sum / otherPatch.uniqueness;
                    bestMatch = selfPatchIndex;
                }
            }
        }

        matches[patchIndex].srcIndex = offset+patchIndex;
        matches[patchIndex].closestMatchIndex = bestMatch;
        matches[patchIndex].sqrDiff = bestMatchSqrDiff;
        matches[patchIndex].normalizedDiff = bestMatchNormDiff;
    }
    //std::cout << "Average number of compares per patch: " << totalNumCompares / (float)count << std::endl;
    #else
    for (unsigned patchIndex = 0; patchIndex < count; patchIndex++) {
        unsigned bestMatch = -1;
        float bestMatchSqrDiff = 1e30f;
        float bestMatchNormDiff = 1e30f;

        const SiftDescriptor &otherPatch = descriptors[patchIndex];

        for (unsigned selfPatchIndex = 0; selfPatchIndex < m_descriptors.size(); selfPatchIndex++) {

            const SiftDescriptor &thisPatch = m_descriptors[selfPatchIndex];

            float sum = thisPatch.computeDifference(otherPatch);

            if (sum < bestMatchSqrDiff) {
                bestMatchSqrDiff = sum;
                bestMatchNormDiff = sum / otherPatch.score;
                bestMatch = selfPatchIndex;
            }
        }

        matches[patchIndex].srcIndex = offset+patchIndex;
        matches[patchIndex].closestMatchIndex = bestMatch;
        matches[patchIndex].sqrDiff = bestMatchSqrDiff;
        matches[patchIndex].normalizedDiff = bestMatchNormDiff;
    }
    #endif
}


void SiftDescriptorDB::findMatchesConstrainedSubrange(MatchConstraint *constraint, const SiftDescriptor **descriptors, unsigned offset, unsigned count) const
{
/*
    unsigned numGridCellsToVisit = 1;
    for (unsigned i = 0; i < m_gridDim; i++)
        numGridCellsToVisit *= 3;
*/
    //unsigned totalNumCompares = 0;

    for (unsigned patchIndex = offset; patchIndex < offset+count; patchIndex++) {
        const SiftDescriptor &otherPatch = *descriptors[patchIndex];
        const unsigned otherPatchCell = otherPatch.projectionIndex % m_grid.size();

/*
        for (unsigned cell = 0; cell < numGridCellsToVisit; cell++) {

            bool validCell = true;

            unsigned cellIndex;
            {
                unsigned bucketsIndices[8];

                unsigned srcCellIndex = otherPatch.projectionIndex % m_grid.size();
                unsigned destrCell = cell;
                for (unsigned i = 0; i < m_gridDim; i++) {
                    int bucket = srcCellIndex % m_gridResolution;
                    int bucketOffset = (int)(destrCell % 3) - 1;

                    bucket += bucketOffset;

                    if ((bucket < 0) || (bucket >= (int)m_gridResolution)) {
                        validCell = false;
                        break;
                    }

                    bucketsIndices[i] = bucket;

                    srcCellIndex /= m_gridResolution;
                    destrCell /= 3;
                }

                if (!validCell)
                    continue;
                cellIndex = 0;
                for (unsigned i = 0; i < m_gridDim; i++) {
                    cellIndex = cellIndex * m_gridResolution + bucketsIndices[m_gridDim-1-i];
                }
            }
*/
        for (unsigned cell = 0; cell < m_lookupOffsets.size(); cell++) {
            unsigned cellIndex = otherPatchCell + m_lookupOffsets[cell];
            if (cellIndex >= m_grid.size())
                continue;

            //todo: this is pointer chasing heavy

            for (unsigned selfPatchIndex = m_grid[cellIndex].start; selfPatchIndex < m_grid[cellIndex].start+m_grid[cellIndex].count; selfPatchIndex++) {
            //for (unsigned selfPatchIndex = 0; selfPatchIndex < m_candidates.size(); selfPatchIndex++) {


                const SiftDescriptor &thisPatch = m_descriptors[selfPatchIndex];

                if (constraint->preMatchTest(patchIndex, selfPatchIndex)) {

                    float sum = thisPatch.computeDifference(otherPatch);

                    constraint->postMatchOperation(patchIndex, selfPatchIndex, sum / otherPatch.uniqueness);
                }
            }
        }
    }
}



void SiftDescriptorDB::findMatches(const SiftDescriptor *descriptors, unsigned count, std::vector<Match> &matches, unsigned batchSize) const
{
    matches.resize(count);
#if 1
    if (count > batchSize) {
        TaskGroup group;
        for (unsigned i = 0; i < count; i+= batchSize) {
            group.add(boost::bind(&SiftDescriptorDB::findMatchesSubrange, this,
                                  descriptors + i, std::min<unsigned>(batchSize, count-i), i, &matches[i]), TaskScheduler::get());
        }
        TaskScheduler::get().waitFor(&group);
    } else
#endif
        findMatchesSubrange(descriptors, count, 0, &matches[0]);

    std::sort(matches.begin(), matches.end());
}


void SiftDescriptorDB::findMatches(const SiftDescriptorDB &other, std::vector<Match> &matches, unsigned batchSize) const
{
    findMatches(&other.m_descriptors[0], other.m_descriptors.size(), matches, batchSize);
}


void SiftDescriptorDB::findMatchesConstrained(MatchConstraint *constraint, unsigned batchSize) const
{
    const unsigned count = constraint->getNumDescriptors();
    const SiftDescriptor **descriptors = constraint->getDescriptors();
#if 1
    if (count > batchSize) {
        TaskGroup group;
        for (unsigned i = 0; i < count; i+= batchSize) {
            group.add(boost::bind(&SiftDescriptorDB::findMatchesConstrainedSubrange, this,
                                  constraint, descriptors, i, std::min<unsigned>(batchSize, count-i)), TaskScheduler::get());
        }
        TaskScheduler::get().waitFor(&group);
    } else
#endif
        findMatchesConstrainedSubrange(constraint, descriptors, 0, count);
}

void SiftDescriptorDB::findMatchesConstrainedAsync(MatchConstraint *constraint, AsyncMatching &asyncMatching, unsigned batchSize) const
{
    const unsigned count = constraint->getNumDescriptors();
    const SiftDescriptor **descriptors = constraint->getDescriptors();
#if 1
    for (unsigned i = 0; i < count; i+= batchSize) {
        asyncMatching.m_group.add(boost::bind(&SiftDescriptorDB::findMatchesConstrainedSubrange, this,
                              constraint, descriptors, i, std::min<unsigned>(batchSize, count-i)), TaskScheduler::get());
    }
#else
    findMatchesConstrainedSubrange(constraint, descriptors, 0, count);
#endif
}



void SiftDescriptorDB::recomputeUniquenessesSubrange(const std::vector<LinAlg::Vector2f> &screenPositions, unsigned offset, unsigned count)
{
    #if 1
    unsigned numGridCellsToVisit = 1;
    for (unsigned i = 0; i < m_gridDim; i++)
        numGridCellsToVisit *= 3;

    for (unsigned patchIndex = offset; patchIndex < offset+count; patchIndex++) {
        //float bestMatchSqrDiff = 1e30f;
        float bestMatchSqrDiff = 0.2f;

        SiftDescriptor &otherPatch = m_descriptors[patchIndex];
        /*
        for (unsigned cell = 0; cell < numGridCellsToVisit; cell++) {

            bool validCell = true;

            unsigned cellIndex;
            {
                unsigned bucketsIndices[8];

                unsigned srcCellIndex = otherPatch.projectionIndex % m_grid.size();
                unsigned destrCell = cell;
                for (unsigned i = 0; i < m_gridDim; i++) {
                    int bucket = srcCellIndex % m_gridResolution;
                    int bucketOffset = (int)(destrCell % 3) - 1;

                    bucket += bucketOffset;

                    if ((bucket < 0) || (bucket >= (int)m_gridResolution)) {
                        validCell = false;
                        break;
                    }

                    bucketsIndices[i] = bucket;

                    srcCellIndex /= m_gridResolution;
                    destrCell /= 3;
                }

                if (!validCell)
                    continue;
                cellIndex = 0;
                for (unsigned i = 0; i < m_gridDim; i++) {
                    cellIndex = cellIndex * m_gridResolution + bucketsIndices[m_gridDim-1-i];
                }
            }
*/
        const unsigned otherPatchCell = otherPatch.projectionIndex % m_grid.size();

        for (unsigned cell = 0; cell < m_lookupOffsets.size(); cell++) {
            unsigned cellIndex = otherPatchCell + m_lookupOffsets[cell];
            if (cellIndex >= m_grid.size())
                continue;

            for (unsigned selfPatchIndex = m_grid[cellIndex].start; selfPatchIndex < m_grid[cellIndex].start+m_grid[cellIndex].count; selfPatchIndex++) {
            //for (unsigned selfPatchIndex = 0; selfPatchIndex < m_candidates.size(); selfPatchIndex++) {

                const SiftDescriptor &thisPatch = m_descriptors[selfPatchIndex];

                if (selfPatchIndex == patchIndex)
                    continue;
                {
                    if ((screenPositions[thisPatch.userData] - screenPositions[otherPatch.userData]).SQRLen() < 1e-6f)
                        continue;
                }

                float sum = thisPatch.computeDifference(otherPatch);

                if (sum < bestMatchSqrDiff) {
                    bestMatchSqrDiff = sum;
                }
            }
        }
        otherPatch.uniqueness = std::max(bestMatchSqrDiff, 1e-20f);
    }
    #else
    for (unsigned patchIndex = offset; patchIndex < offset+count; patchIndex++) {
        //float bestMatchSqrDiff = 1e30f;
        float bestMatchSqrDiff = 20.0f;

        SiftDescriptor &otherPatch = m_descriptors[patchIndex];

        for (unsigned selfPatchIndex = 0; selfPatchIndex < m_descriptors.size(); selfPatchIndex++) {

            const SiftDescriptor &thisPatch = m_descriptors[selfPatchIndex];

            if (selfPatchIndex == patchIndex)
                continue;
            {
                if ((screenPositions[thisPatch.userData] - screenPositions[otherPatch.userData]).SQRLen() < 1e-6f)
                    continue;
            }

            float sum = thisPatch.computeDifference(otherPatch);

            if (sum < bestMatchSqrDiff) {
                bestMatchSqrDiff = sum;
            }
        }

        otherPatch.score = std::max(bestMatchSqrDiff, 1e-20f);
    }
    #endif
}

void SiftDescriptorDB::recomputeUniquenesses(const std::vector<LinAlg::Vector2f> &screenPositions)
{
#if 1
    if (m_descriptors.size() > 200) {
        TaskGroup group;
        for (unsigned i = 0; i < m_descriptors.size(); i+= 200) {
            group.add(boost::bind(&SiftDescriptorDB::recomputeUniquenessesSubrange, this,
                                  screenPositions, i, std::min<unsigned>(200, m_descriptors.size()-i)), TaskScheduler::get());
        }
        TaskScheduler::get().waitFor(&group);
    } else
#endif
        recomputeUniquenessesSubrange(screenPositions, 0, m_descriptors.size());

/*
    for (unsigned i = 0; i < m_descriptors.size(); i++) {
        m_descriptors[i].uniqueFeaturePointID = -1;
    }
    unsigned ID = 0;
    for (unsigned i = 0; i < m_descriptors.size(); i++) {
        if (m_descriptors[i].uniqueFeaturePointID != (unsigned)-1)
            m_descriptors[i].uniqueFeaturePointID = ID++;

        for (unsigned j = i+1; i < m_descriptors.size(); j++) {
                if ((screenPositions[i] - screenPositions[j]).SQRLen() < 1e-6f)
                    m_descriptors[j].uniqueFeaturePointID = i;
        }
    }
    */
}


}
