/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef SIFTDESCRIPTORDB_H
#define SIFTDESCRIPTORDB_H

#include <stdint.h>
#include <vector>
#include "../tools/LinAlg.h"

#include "../tools/TaskScheduler.h"


namespace SFM {

/**
 * @brief Stores a list of feature descriptors.
 * @details The feature descriptor are stored in a hyper grid
 * structure that allows fast matching with feature descriptors
 * from other @ref SFM::SiftDescriptorDB. Usually every @ref SFM::Frame
 * stores its feature descriptors in such a data base.
 * @ingroup SFMBackend_Group
 */
class SiftDescriptorDB
{
    public:
        SiftDescriptorDB(unsigned gridDim, unsigned gridRes);
        ~SiftDescriptorDB();

        /// Sift feature point descriptor
        struct SiftDescriptor {
            /// Carries user defined data.
            unsigned userData;
            /// How unique the feature descriptor is in its own image. Used to normalize matching scores.
            float uniqueness;
            /// Hyper grid cell index into which this feature descriptor belongs
            unsigned projectionIndex;
            /// Reciprocal of the feature descriptors length.
            float rcpLength;
            /// The 128 dimensional descriptor vector stored as 128 bytes.
            uint32_t descriptor[32] __attribute__((aligned (16)));

            float computeDifference(const SiftDescriptor &other) const;
        };

        void computeProjectionIndices(SiftDescriptor *descriptors, unsigned count) const;
        void compile(SiftDescriptor *descriptors, unsigned count);

        struct Match {
            unsigned srcIndex;
            unsigned closestMatchIndex;
            float sqrDiff;
            float normalizedDiff;
            inline bool operator<(const Match &other) const { return normalizedDiff < other.normalizedDiff; }
        };


        struct MatchConstraint {
            virtual unsigned getNumDescriptors() const = 0;
            virtual const SiftDescriptor **getDescriptors() = 0;
            virtual bool preMatchTest(unsigned sourceIndex, unsigned destinationIndex) = 0;
            virtual void postMatchOperation(unsigned sourceIndex, unsigned destinationIndex, float normalizedMatchScore) = 0;
        };

        struct AsyncMatching {
            TaskGroup m_group;
            void waitFor() { TaskScheduler::get().waitFor(&m_group); }
        };

        void findMatchesConstrained(MatchConstraint *constraint, unsigned batchSize = 200) const;
        void findMatchesConstrainedAsync(MatchConstraint *constraint, AsyncMatching &asyncMatching, unsigned batchSize = 200) const;

        void findMatches(const SiftDescriptor *descriptors, unsigned count, std::vector<Match> &matches, unsigned batchSizes = 200) const;
        void findMatches(const SiftDescriptorDB &other, std::vector<Match> &matches, unsigned batchSize = 200) const;

        inline unsigned getNumDescriptors() const { return m_descriptors.size(); }
        const SiftDescriptor &getDescriptor(unsigned index) const { return m_descriptors[index]; }

        void setUserData(unsigned index, unsigned userData) { m_descriptors[index].userData = userData; }


        void recomputeUniquenesses(const std::vector<LinAlg::Vector2f> &screenPositions);

    protected:
        void computeProjectionIndicesSubrange(SiftDescriptor *descriptors, unsigned count) const;
        void findMatchesSubrange(const SiftDescriptor *descriptors, unsigned count, unsigned offset, Match *matches) const;
        void findMatchesConstrainedSubrange(MatchConstraint *constraint, const SiftDescriptor **descriptors, unsigned offset, unsigned count) const;
        void recomputeUniquenessesSubrange(const std::vector<LinAlg::Vector2f> &screenPositions, unsigned offset, unsigned count);


        struct GridEntry {
            unsigned start;
            unsigned count;
        };

        struct SiftGridBasis
        {
            LinAlg::Vector<128, float> offset[8];
            LinAlg::Vector<128, float> direction[8];
        };

        SiftGridBasis m_basis;
        int16_t m_projectionBasis[8*128] __attribute__((aligned (16)));
        float m_projectionBasisScales[8];
        float m_projectionOffsets[8];

        unsigned m_gridDim;
        unsigned m_gridResolution;

        std::vector<SiftDescriptor> m_descriptors;
        std::vector<GridEntry> m_grid;



        std::vector<int> m_lookupOffsets;


};

}

#endif // SIFTDESCRIPTORDB_H
