/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "TrackObservation.h"

#include "Frame.h"
#include "Track.h"
//#include "LensDistortionGrid.h"
#include "InternalCameraCalibration.h"


namespace SFM {

TrackObservation::TrackObservation() : m_observationGridListEntry(*this)
{
    m_track = NULL;
    m_frame = NULL;
    m_frameFeaturePointIndex = -1;
    m_baObsHandle = -1;
    m_faulty = false;
    m_referenceObsHomographyValid = false;
    m_referenceAlignmentError = -1.0f;
}

TrackObservation::~TrackObservation()
{
    //dtor
}


bool TrackObservation::isReferenceObs() const
{
    return m_track->getReferenceObservation() == this;
}


void TrackObservation::setup(Track *track, Frame *frame, unsigned frameFeaturePointIndex)
{
    m_track = track;
    m_frame = frame;
    m_frameFeaturePointIndex = frameFeaturePointIndex;
    m_distortedScreenSpacePosition[0] = m_frame->getFeaturePoint(m_frameFeaturePointIndex).distortedX;
    m_distortedScreenSpacePosition[1] = m_frame->getFeaturePoint(m_frameFeaturePointIndex).distortedY;
    m_undistortedScreenSpacePosition[0] = m_frame->getFeaturePoint(m_frameFeaturePointIndex).undistortedX;
    m_undistortedScreenSpacePosition[1] = m_frame->getFeaturePoint(m_frameFeaturePointIndex).undistortedY;
    m_weight = 1.0f / std::sqrt(m_frame->getFeaturePoint(m_frameFeaturePointIndex).size);
//    m_weight = std::sqrt(std::sqrt(std::max(m_frame->getFeaturePoint(m_frameFeaturePointIndex).locationPrecision, 1e-10f)) * 100.0f);

    m_referenceObsHomographyValid = false;

    m_overallWeight = m_weight * m_track->getTrackWeight();

    ObservationGrid &grid = m_frame->getObservationGrid();

    m_observationGridListEntry.appendToList(&grid.getCell(m_distortedScreenSpacePosition).observations);
}

void TrackObservation::activateBA()
{
    assert(m_baObsHandle == (unsigned)-1);

    m_baObsHandle = m_track->getBA()->addObservation(m_frame->getCamera().getBAHandle(), m_track->getBAHandle());

    m_track->getBA()->setObservation(m_baObsHandle, m_distortedScreenSpacePosition, m_overallWeight);
}

void TrackObservation::deactivateBA()
{
    m_track->getBA()->removeObservation(m_baObsHandle);
    m_baObsHandle = -1;
}

float TrackObservation::computeSQRScreenSpaceError() const
{
    LinAlg::Vector4f projScreenPos = m_frame->getCamera().getProjectionViewMatrix() * m_track->getLastWSPositionEstimate();
    float x = projScreenPos[0] * (1.0f / projScreenPos[3]);
    float y = projScreenPos[1] * (1.0f / projScreenPos[3]);

    const BundleAdjustment::RadialDistortionParametrization &distortionParams =
            m_frame->getCamera().getInternalCalibration()->getRadialDistortion();

    switch (distortionParams.type) {
        case config::BundleAdjustmentStructureConfig::RadialDistortionType::NoRadialDistortion:
        break;
        case config::BundleAdjustmentStructureConfig::RadialDistortionType::Polynomial_234: {
            float r2 = x*x + y*y;
            float r3 = r2 * std::sqrt(r2);
            float r4 = r2*r2;
            float R = 1.0f + r2 * distortionParams.polynomial234.kappa[0] +
                             r3 * distortionParams.polynomial234.kappa[1] +
                             r4 * distortionParams.polynomial234.kappa[2];
            x *= R;
            y *= R;
        } break;
        default:
            throw std::runtime_error("Unhandled distortion type!");
    }

    float dx = x - m_distortedScreenSpacePosition[0];
    float dy = y - m_distortedScreenSpacePosition[1];

    return dx*dx + dy*dy;
}

void TrackObservation::markFaulty()
{
    m_faulty = true;
    if (m_baObsHandle != (unsigned)-1) {
        m_track->getBA()->removeObservation(m_baObsHandle);
        m_baObsHandle = -1;
    }
    m_track->obsChanged();
}

inline bool fastMathIsFinite(float f)
{
    union {
        float f;
        unsigned i;
    } f2i;

    f2i.f = f;

    return ((f2i.i >> 23) & 0xFF) != 0xFF;
}

void TrackObservation::setScreenSpacePosition(const LinAlg::Vector2f &screenPos)
{
#if 0
    const Frame::ImageFeaturePoint &FP = m_frame->getFeaturePoint(m_frameFeaturePointIndex);
    float sqrDist = (screenPos - LinAlg::Fill(FP.x, FP.y)).SQRLen();
    if (std::sqrt(sqrDist) > 5.0f / 2000) {
        std::cout << "TrackObservation::setScreenSpacePosition - sqrDist == " << sqrDist  << std::endl;
    }
    assert(sqrDist < 1.0f);
#endif

    throw std::runtime_error("Not supported ATM!");

    if ((!fastMathIsFinite(screenPos[0])) || (!fastMathIsFinite(screenPos[1]))) {
        std::cout << "warning: ignoring new observation position: " << (std::string)screenPos << std::endl;
        return;
    }

    m_distortedScreenSpacePosition = screenPos;
    if (m_baObsHandle != (unsigned)-1) {
        m_track->getBA()->setObservation(m_baObsHandle, m_distortedScreenSpacePosition, m_overallWeight);
    }
}

void TrackObservation::trackWeightChanged()
{
    m_overallWeight = m_weight * m_track->getTrackWeight();
    if (m_baObsHandle != (unsigned)-1) {
        m_track->getBA()->setObservation(m_baObsHandle, m_distortedScreenSpacePosition, m_overallWeight);
    }
}

void TrackObservation::setReferenceObsScreenToThisScreen(const LinAlg::Matrix3x3f &mat, float alignmentError)
{
    m_referenceObsScreenToThisScreen = mat;
    m_referenceObsHomographyValid = true;
    m_referenceAlignmentError = alignmentError;
    m_track->setNeedsNormalUpdate();
}

void TrackObservation::recomputeUndistortedLocation()
{
    m_undistortedScreenSpacePosition[0] = m_frame->getFeaturePoint(m_frameFeaturePointIndex).undistortedX;
    m_undistortedScreenSpacePosition[1] = m_frame->getFeaturePoint(m_frameFeaturePointIndex).undistortedY;
}


}
