/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef TRACKOBSERVATION_H
#define TRACKOBSERVATION_H

#include "../tools/LinAlg.h"
#include "../tools/LinkedList.hpp"
#include "BundleAdjustment/BundleAdjustment.h"

namespace SFM {

class Track;
class Frame;

/**
 * @brief Represents an observation of a track by a frame/camera.
 * @details Can become faulty (be identified as a wrong match).
 * @ingroup SFMBackend_Group
 */
class TrackObservation
{
    public:
        TrackObservation();
        ~TrackObservation();

        void setup(Track *track, Frame *frame, unsigned frameFeaturePointIndex);

        void activateBA();
        void deactivateBA();

        float computeSQRScreenSpaceError() const;

        bool isFaulty() const { return m_faulty; }
        void markFaulty();

        bool activeInBA() const { return m_baObsHandle != (unsigned)-1; }

        inline Frame *getFrame() { return m_frame; }
        inline const Frame *getFrame() const { return m_frame; }
        inline unsigned getFrameFeaturePointIndex() const { return m_frameFeaturePointIndex; }

        inline const LinAlg::Vector2f &getDistortedScreenSpacePosition() const { return m_distortedScreenSpacePosition; }
        inline const LinAlg::Vector2f &getUndistortedScreenSpacePosition() const { return m_undistortedScreenSpacePosition; }
        void setScreenSpacePosition(const LinAlg::Vector2f &screenPos);

        inline float getOverallWeight() const { return m_overallWeight; }
        inline float getWeight() const { return m_weight; }

        void trackWeightChanged();

        inline Track *getTrack() { return m_track; }
        inline const Track *getTrack() const { return m_track; }

        bool isReferenceObs() const;
        void invalidateReferenceObsHomography() { m_referenceObsHomographyValid = false; }
        inline const LinAlg::Matrix3x3f &getReferenceObsScreenToThisScreen() const { return m_referenceObsScreenToThisScreen; }
        void setReferenceObsScreenToThisScreen(const LinAlg::Matrix3x3f &mat, float alignmentError);
        inline bool referenceObsHomographyValid() const { return m_referenceObsHomographyValid; }
        inline float getReferenceAlignmentError() const { return m_referenceAlignmentError; }

        void recomputeUndistortedLocation();

    protected:
        bool m_faulty;
        Track *m_track;
        Frame *m_frame;
        unsigned m_frameFeaturePointIndex;
        BundleAdjustment::ObservationHandle m_baObsHandle;
        LinAlg::Vector2f m_distortedScreenSpacePosition;
        LinAlg::Vector2f m_undistortedScreenSpacePosition;

        bool m_referenceObsHomographyValid;
        LinAlg::Matrix3x3f m_referenceObsScreenToThisScreen;
        float m_referenceAlignmentError;

        float m_weight;
        float m_overallWeight;

        LinkedListEntry<TrackObservation> m_observationGridListEntry;

};

}

#endif // TRACKOBSERVATION_H
