/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef VIEWMATRIXESTIMATOR_H
#define VIEWMATRIXESTIMATOR_H

#include "../tools/LinAlg.h"
#include <vector>

namespace SFM {

/**
 * @brief Ransac based estimation of a view matrix from a set of observed tracks.
 * @ingroup SFMBackend_Group
 */
class ViewMatrixEstimator
{
    public:
        struct ControlPoint {
            unsigned index[2];
            LinAlg::Vector4f worldSpacePosition;
            LinAlg::Vector2f screenSpacePosition;
            float rcpSqrScreenSize;
            float normalizedDifference;
        };

        ViewMatrixEstimator();
        ~ViewMatrixEstimator();

        void estimate(const LinAlg::Matrix3x4f &internalCalibration, const ControlPoint *controlPoints, float *probs, unsigned count, unsigned iterations, LinAlg::Matrix4x4f &bestViewMatrix, float &bestScore);
    protected:
        std::vector<float> m_cumulativeProbabilities;
        std::vector<float> m_probabilities;
        LinAlg::Matrix3x4f m_internalCalibration;


        void buildProbabilities(float lambda, const ControlPoint *controlPoints, unsigned count);
        unsigned pickMatch(float frac) const;


        void doRansac(const ControlPoint *controlPoints, unsigned count, unsigned seed, unsigned iters, LinAlg::Matrix4x4f *dstViewMatrix, float *dstScore);
        void computeProbabilityOfFourProjectionMatrices(const ControlPoint *controlPoints, unsigned count, LinAlg::Matrix3x4f *projectionMatrices, float *probs);
};

}

#endif // VIEWMATRIXESTIMATOR_H
