/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef BUNDLEADJUSTMENTCONFIG_H
#define BUNDLEADJUSTMENTCONFIG_H

#include <stdint.h>

namespace SFM {
namespace config {

/**
 * @brief Bundle adjustment parameters
 *
 * @ingroup SFMBackend_Group
 */
struct BundleAdjustmentParameterConfig
{
    /// The damping parameter to start with.
    float LevenbergMarquardt_InitialLambda = 1e-3f;
    /// Factor by which the damping parameter gets increased for every misstep
    float LevenbergMarquardt_LambdaIncreaseFactor = 5.0f;
    /// Factor by which the damping parameter gets decreased for every successfull step
    float LevenbergMarquardt_LambdaDecreaseFactor = 0.2f;
    /// The minimum bound on the damping parameter
    float LevenbergMarquardt_MinLambda = 1e-10f;
    /// Maximal value of the damping parameter for which convergence is assumed
    float LevenbergMarquardt_MaxLambdaForConvergence = 1e9f;
    /// Maximal number of PCGD iterations per LM iteration
    unsigned PCGD_MaxIterations = 30;
    /// Threshold when to abort PCGD
    float PCGD_RtimesZThresholdFactor = 1e-5f;
    /// How much debug output to print
    unsigned verbatimLevel = 0;
};

/**
 * @brief Bundle adjustment structure configuration
 *
 * @ingroup SFMBackend_Group
 */
struct BundleAdjustmentStructureConfig
{
    enum class InternalCalibrationType : uint8_t {
        FullCalibration = 0,    ///< All parameters of the internal calibration may be adjusted by BA
        SharedFocalLength = 1,  ///< All parameters of the internal calibration may be adjusted by BA, but the focal lengths may only be adjusted together
        Fixed = 2               ///< No parameters of the internal calibration may be adjusted by BA
    };
    /// Defines to what degree the internal calibration is to be tweaked by the bundle adjustment.
    InternalCalibrationType internalCalibrationType = InternalCalibrationType::FullCalibration;

    enum class RadialDistortionType : uint8_t {
        NoRadialDistortion = 0, ///< Assume no radial distortion
        Polynomial_234 = 1      ///< Use a polynomial model with the koefficients kappa_2 through kappa_4
    };
    /// Radial distortion model. For a fixed internal calibration, the distortion parameters is also fixed.
    RadialDistortionType radialDistortionType = RadialDistortionType::Polynomial_234;
};

}
}

#endif // BUNDLEADJUSTMENTCONFIG_H
