/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef FEATUREMATCHINGCONFIG_H
#define FEATUREMATCHINGCONFIG_H

#include <stdint.h>
#include <vector>


namespace SFM {
namespace config {

/**
 * @brief Parameters of the feature matching step
 *
 * @ingroup SFMBackend_Group
 */

struct FeatureMatchingConfig
{
    /// Number of matching operations per batch. Batches are distributed among the threads in the thread pool.
    unsigned matchingMultithreadingBatchSize = 200;
    /// Maximum value for a feature points uniqueness in case no similar feature point is found.
    float maxFeaturePointUniqueness = 0.2f;


    /// Maximal matching difference for the feature point matching of the initial image pair
    float initialImagePairMatchingThreshold = 0.6f;

    /// Number of horizontal subdivisions of the grid that is used to evaluate the need to match more feature points (and also SubsetBA subset selection).
    unsigned frameGridSubdivisions = 10;
    struct FrameNewCameraMatchingConfig {
        /// Matching threshold for the initial track vs. image feature point matching
        float maxNormalizedMatchDiff = 0.5f;
        /// How many matches have to be found initially for a RANSAC based estimation of the cameras pose to be attempted.
        unsigned minInitialMatchCount = 50;
        /// How many RANSAC iterations to run with the initial matches
        unsigned numInitialRansacIterations = 30000;

        /// Minimal score of the best RANSAC result to continue with the boostrapping
        float minInitialRansacScore = 50.0f;
        /// Minimal number of matches of the bootstrapping location constrained matching
        unsigned minBootstrappingMatchCount = 80;
        /// Number of RANSAC iterations to perform on the bootstrapping matches
        unsigned numSecondaryRansacIterations = 1000;
        /// Minimal score of the best RANSAC result from the boostrapping
        float minSecondaryRansacScore = 80.0f;
        /// Minimal score of each observation to be accepted
        float minNewObsScore = 0.1f;
        /// Minimal number of accepted observations to accept the new camera/frame.
        unsigned minAcceptedObs = 80;
    };
    /// Parameters for adding new cameras
    FrameNewCameraMatchingConfig frameNewCameraMatchingConfig;

    struct FrameNewObservationsMatchingConfig {
        /// (Soft) maximal squared distance error
        float maxSqrDistanceError;
        /// (Soft) maximal matching difference
        float maxNormalizedMatchDiff;
        /// Minimal score of a match to be accepted
        float minScoreThreshold;
        FrameNewObservationsMatchingConfig(float maxDist, float maxDiff, float minScoreThres) :
            maxSqrDistanceError(maxDist), maxNormalizedMatchDiff(maxDiff), minScoreThreshold(minScoreThres) { }
    };
    /// Parameters for the bootstrapping of a new camera (finding additional observations that link the new camera to the reconstruction)
    FrameNewObservationsMatchingConfig frameBoostrappingMatchingConfig = FrameNewObservationsMatchingConfig(64.0f*64.0f, 0.9f, 0.05f);
    /// Parameters for finding new observations for already reconstructed cameras
    FrameNewObservationsMatchingConfig frameNewObservationsMatchingConfig = FrameNewObservationsMatchingConfig(8.0f*8.0f, 0.6f, 0.05f);

    struct FrameNewTracksMatchingConfig {
        /// (Soft) maximal distance error
        float maxDistanceError = 16.0f;
        /// (Soft) maximal matching difference
        float maxNormalizedMatchDiff = 0.9f;

        /// Maximal number of active observations per cell when to stop looking for new tracks for that cell
        unsigned maxNumActiveObsPerCell = 32;
        /// Minimal uniqueness of an observation to use it for matching
        float minObservationUniqueness = 0.02f;
        /// Maximal number of observations to use for finding new tracks per trial and cell.
        unsigned maxObservationsTestedPerCellNum = 20ul;

        /// Minimal score of a match to be accepted
        float minScoreThreshold = 0.1f;
    };
    /// Parameters for finding new tracks
    FrameNewTracksMatchingConfig frameNewTracksMatchingConfig;

    /// How much debug output to print
    unsigned verbatimLevel = 0;
};

}
}

#endif // FEATUREMATCHINGCONFIG_H
