/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @file
 * @author Andreas Ley
 */

#ifndef _AABB_H_
#define _AABB_H_


#include "LinAlg.h"

namespace LinAlg {

/** @addtogroup Codebase_Group
 *  @{
 */

template <int dimension, typename type>
class AABB
{
    public:
        typedef Vector<dimension, type> VectorType;

        AABB() { }
        AABB(const VectorType &v) { m_min = m_max = v; }

        inline const AABB& operator=(const AABB &rhs) {
            m_min = rhs.m_min;
            m_max = rhs.m_max;
            return *this;
        }
        inline const AABB& operator=(const VectorType &rhs) {
            m_min = rhs;
            m_max = rhs;
            return *this;
        }

        inline const AABB& operator+=(const AABB &rhs) {
            for (unsigned i = 0; i < dimension; i++)
                m_min[i] = std::min(m_min[i], rhs.m_min[i]);
            for (unsigned i = 0; i < dimension; i++)
                m_max[i] = std::max(m_max[i], rhs.m_max[i]);
            return *this;
        }
        inline const AABB& operator+=(const VectorType &rhs) {
            for (unsigned i = 0; i < dimension; i++) {
                m_min[i] = std::min(m_min[i], rhs[i]);
                m_max[i] = std::max(m_max[i], rhs[i]);
            }
            return *this;
        }

        inline const AABB& operator+(const AABB &rhs) const {
            AABB aabb(*this);
            aabb += rhs;
            return aabb;
        }
        inline const AABB& operator+(const VectorType &rhs) const {
            AABB aabb(*this);
            aabb += rhs;
            return aabb;
        }

        inline bool contains(const VectorType &point) const {
            for (unsigned i = 0; i < dimension; i++) {
                if (m_min[i] > point[i])
                    return false;
                if (m_max[i] < point[i])
                    return false;
            }
            return true;
        }

        inline bool contains(const AABB &rhs) const {
            for (unsigned i = 0; i < dimension; i++) {
                if (m_min[i] > rhs.m_min[i])
                    return false;
                if (m_max[i] < rhs.m_max[i])
                    return false;
            }
            return true;
        }

        inline bool intersects(const AABB &rhs) const {
            for (unsigned i = 0; i < 3; i++) {
                if (m_min[i] > rhs.m_max[i])
                    return false;
                if (m_max[i] < rhs.m_min[i])
                    return false;
            }
            return true;
        }

        inline const VectorType& min() const { return m_min; }
        inline const VectorType& max() const { return m_max; }

        inline VectorType& min() { return m_min; }
        inline VectorType& max() { return m_max; }

        inline VectorType computeExtends() const { return m_max - m_min; }
        inline VectorType computeCenter() const { return (m_max + m_min) * 0.5; }

    protected:
        VectorType m_min;
        VectorType m_max;
};


typedef AABB<2, float> AABB2f;
typedef AABB<3, float> AABB3f;


/// @}

}


#endif // _AABB_H_
