#ifndef QUATERNION_H
#define QUATERNION_H

#include <cmath>
#include "LinAlg.h"

template<typename Type>
class Quaternion
{
    public:
        Quaternion() {
            m_rijk[0] =
            m_rijk[1] =
            m_rijk[2] =
            m_rijk[3] = 0.0f;
        }

        Quaternion(Type real, Type i, Type j, Type k) {
            m_rijk[0] = real;
            m_rijk[1] = i;
            m_rijk[2] = j;
            m_rijk[3] = k;
        }

        static Quaternion<Type> fromAxisAngle(Type angle, Type x, Type y, Type z) {
            Type s = std::sin(angle/2);
            Type c = std::cos(angle/2);
            return Quaternion<Type>(c, s*x, s*y, s*z);
        }

        Type &operator[](unsigned index) { return m_rijk[index]; }
        const Type &operator[](unsigned index) const { return m_rijk[index]; }

        Quaternion<Type> operator+(const Quaternion<Type> &rhs) const {
            return Quaternion<Type>(
                m_rijk[0] + rhs.m_rijk[0],
                m_rijk[1] + rhs.m_rijk[1],
                m_rijk[2] + rhs.m_rijk[2],
                m_rijk[3] + rhs.m_rijk[3]
            );
        }
        const Quaternion<Type> &operator+=(const Quaternion<Type> &rhs) const {
            m_rijk[0] += rhs.m_rijk[0];
            m_rijk[1] += rhs.m_rijk[1];
            m_rijk[2] += rhs.m_rijk[2];
            m_rijk[3] += rhs.m_rijk[3];
            return *this;
        }


        Quaternion<Type> operator-(const Quaternion<Type> &rhs) const {
            return Quaternion<Type>(
                m_rijk[0] - rhs.m_rijk[0],
                m_rijk[1] - rhs.m_rijk[1],
                m_rijk[2] - rhs.m_rijk[2],
                m_rijk[3] - rhs.m_rijk[3]
            );
        }
        const Quaternion<Type> &operator-=(const Quaternion<Type> &rhs) const {
            m_rijk[0] -= rhs.m_rijk[0];
            m_rijk[1] -= rhs.m_rijk[1];
            m_rijk[2] -= rhs.m_rijk[2];
            m_rijk[3] -= rhs.m_rijk[3];
            return *this;
        }

        Quaternion<Type> operator*(const Quaternion<Type> &rhs) const {
            return Quaternion<Type>(
                m_rijk[0]*rhs.m_rijk[0] - m_rijk[1]*rhs.m_rijk[1] - m_rijk[2]*rhs.m_rijk[2] - m_rijk[3]*rhs.m_rijk[3],
                m_rijk[0]*rhs.m_rijk[1] + m_rijk[1]*rhs.m_rijk[0] + m_rijk[2]*rhs.m_rijk[3] - m_rijk[3]*rhs.m_rijk[2],
                m_rijk[0]*rhs.m_rijk[2] - m_rijk[1]*rhs.m_rijk[3] + m_rijk[2]*rhs.m_rijk[0] + m_rijk[3]*rhs.m_rijk[1],
                m_rijk[0]*rhs.m_rijk[3] + m_rijk[1]*rhs.m_rijk[2] - m_rijk[2]*rhs.m_rijk[1] + m_rijk[3]*rhs.m_rijk[0]
            );
        }

        Quaternion<Type> versorInvert() const {
            return Quaternion<Type>(
                m_rijk[0],
                -m_rijk[1],
                -m_rijk[2],
                -m_rijk[3]
            );
        }

        Quaternion<Type> normalized() const {
            float rcpLen = Type(1) / std::sqrt(m_rijk[0]*m_rijk[0] +
                                               m_rijk[1]*m_rijk[1] +
                                               m_rijk[2]*m_rijk[2] +
                                               m_rijk[3]*m_rijk[3]);
            return Quaternion<Type>(
                m_rijk[0] * rcpLen,
                m_rijk[1] * rcpLen,
                m_rijk[2] * rcpLen,
                m_rijk[3] * rcpLen
            );
        }

        LinAlg::Matrix<3, 3, Type> versorToRotationMatrix() const {
            LinAlg::Matrix<3, 3, Type> result(LinAlg::Matrix<3, 3, Type>::NO_INITIALIZATION);

            result[0][0] = m_rijk[0]*m_rijk[0] + m_rijk[1]*m_rijk[1] - m_rijk[2]*m_rijk[2] - m_rijk[3]*m_rijk[3];
            result[0][1] = Type(2) * (m_rijk[1]*m_rijk[2] - m_rijk[0]*m_rijk[3]);
            result[0][2] = Type(2) * (m_rijk[1]*m_rijk[3] + m_rijk[0]*m_rijk[2]);

            result[1][0] = Type(2) * (m_rijk[1]*m_rijk[2] + m_rijk[0]*m_rijk[3]);
            result[1][1] = m_rijk[0]*m_rijk[0] - m_rijk[1]*m_rijk[1] + m_rijk[2]*m_rijk[2] - m_rijk[3]*m_rijk[3];
            result[1][2] = Type(2) * (m_rijk[2]*m_rijk[3] - m_rijk[0]*m_rijk[1]);

            result[2][0] = Type(2) * (m_rijk[1]*m_rijk[3] - m_rijk[0]*m_rijk[2]);
            result[2][1] = Type(2) * (m_rijk[2]*m_rijk[3] + m_rijk[0]*m_rijk[1]);
            result[2][2] = m_rijk[0]*m_rijk[0] - m_rijk[1]*m_rijk[1] - m_rijk[2]*m_rijk[2] + m_rijk[3]*m_rijk[3];

            return result;
        }

    protected:
        Type m_rijk[4];
};

typedef Quaternion<float> Quaternionf;
typedef Quaternion<double> Quaterniond;

#endif // QUATERNION_H
