/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _SVD_HPP_
#define _SVD_HPP_

/**
 * @file
 * @author Andreas Ley
 */


#include "HouseholderReduction.hpp"

namespace LinAlg {

/**
 * @ingroup Codebase_Group
 */
namespace SVD {


/** @addtogroup Codebase_Group
 *  @{
 */


template<typename SourceMatrixType, unsigned Dim, typename Type>
void computeRightSingularVectorsForRealMatrix(const SourceMatrixType &source, Vector<Dim, Type> &singularValues, Matrix<Dim, Dim, Type> &rightSingularVectors)
{
    Matrix<Dim, Dim, Type> product(Matrix<Dim, Dim, Type>::NO_INITIALIZATION);
    for (unsigned i = 0; i < Dim; i++)
        for (unsigned j = i; j < Dim; j++) {
            Type sum = Type(0);
            for (unsigned k = 0; k < source.size(); k++)
                sum += source[k][i] * source[k][j];
            product[i][j] = product[j][i] = sum;
        }

    try {
        computeEigenValues<Dim, Type>(product, singularValues, &rightSingularVectors);
    } catch(...) {
        /*
        std::cout << "Error for matrix: " << std::endl;
        for (unsigned i = 0; i < Dim; i++) {
            for (unsigned j = 0; j < Dim; j++) {
                std::cout << " " << product[i][j];
            }
            if (i < Dim-1)
                std::cout << ";" << std::endl;
        }
*/
        throw;
    }
    for (unsigned i = 0; i < Dim; i++)
        singularValues[i] = std::sqrt(std::max<Type>(0.0, singularValues[i]));
}

/**
 * @ingroup Codebase_Group
 */
template<unsigned Rows, unsigned Dim, typename Type>
void computeRightSingularVectorsForRealMatrixFixedRows(const Matrix<Rows, Dim, Type> &source, Vector<Dim, Type> &singularValues, Matrix<Dim, Dim, Type> &rightSingularVectors)
{
    Matrix<Dim, Dim, Type> product;
    for (unsigned i = 0; i < Dim; i++)
        for (unsigned j = i; j < Dim; j++) {
            Type sum = Type(0);
            for (unsigned k = 0; k < Rows; k++)
                sum += source[k][i] * source[k][j];
            product[i][j] = product[j][i] = sum;
        }

    try {
        computeEigenValues<Dim, Type>(product, singularValues, &rightSingularVectors);
    } catch(...) {
        /*
        std::cout << "Error for matrix: " << std::endl;
        for (unsigned i = 0; i < Dim; i++) {
            for (unsigned j = 0; j < Dim; j++) {
                std::cout << " " << product[i][j];
            }
            if (i < Dim-1)
                std::cout << ";" << std::endl;
        }
        */
        throw;
    }
    for (unsigned i = 0; i < Dim; i++)
        singularValues[i] = std::sqrt(std::max<Type>(0.0, singularValues[i]));
}

/// @}

}

}


#endif // _SVD_HPP_
