/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _TRIDIAGONALSYMMETRIC_HPP_
#define _TRIDIAGONALSYMMETRIC_HPP_

/**
 * @file
 * @author Andreas Ley
 */

#include "LinAlg.h"
#include <stdexcept>
#include <limits>
#include <cmath>
#include <math.h>

namespace LinAlg {


/** @addtogroup Codebase_Group
 *  @{
 */


template<typename T>
inline T sqr(const T &x) { return x*x; }

template<unsigned Dim, typename Type>
class TriDiagonalSymmetricMatrix
{
    public:
        const Vector<Dim, Type> &Diagonal() const { return m_diagonal; }
        Vector<Dim, Type> &Diagonal() { return m_diagonal; }

        const Vector<Dim-1, Type> &OffDiagonal() const { return m_offDiagonal; }
        Vector<Dim-1, Type> &OffDiagonal() { return m_offDiagonal; }

        Matrix<Dim, Dim, Type> inflate() const {
            Matrix<Dim, Dim, Type> result;

            for (unsigned i = 0; i < Dim; i++)
                result[i][i] = m_diagonal[i];
            for (unsigned i = 0; i < Dim-1; i++)
                result[i+1][i] = result[i][i+1] = m_offDiagonal[i];

            return result;
        }

        // Adapted from Numerical Recipes
        template<bool CompEigenVectors>
        void computeEigenValues(Vector<Dim, Type> &eigenValues, Matrix<Dim, Dim, Type> *eigenVectors = NULL) {
            const Type eps = std::numeric_limits<Type>::epsilon();
            Vector<Dim, Type> d = m_diagonal;
            Vector<Dim, Type> e = m_offDiagonal.AddHom(Type(0));

            for (unsigned l = 0; l < Dim; l++) {
                unsigned iteration = 0;
                unsigned m;
                do {
                    for (m = l; m < Dim-1; m++) {
                        Type dd = std::abs(d[m])+std::abs(d[m+1]);
                        if (std::abs(e[m]) < eps*dd)
                            break;
                    }
                    if (m != l) {
                        if (iteration++ > 50) {
                                /*
                            std::cout << l << "  " << m << std::endl;
                            std::cout << (std::string) d << "  " << (std::string) e << std::endl;
                                */
                            throw std::runtime_error("Too many iterations!");
                        }
                        Type g = (d[l+1]-d[l]) / (Type(2)*e[l]);
                        Type r = pythag(g, Type(1));
                        g = d[m] - d[l] + e[l] / (g + copy_sign(r, g));

                        Type s = Type(1);
                        Type c = Type(1);
                        Type p = Type(0);
                        int i;
                        for (i = m-1; i >= (int)l; i--) {
                            Type f = s * e[i];
                            Type b = c * e[i];
                            Type r = pythag(f, g);
                            e[i+1] = r;
                            if (r == Type(0)) {
                                d[i+1] -= p;
                                e[m] = Type(0);
                                break;
                            }
                            s = f / r;
                            c = g / r;
                            g = d[i+1] - p;
                            r = (d[i] - g) * s + Type(2) * c * b;
                            p = s*r;
                            d[i+1] = g + p;
                            g = c * r - b;
                            if (CompEigenVectors) {
                                for (unsigned k = 0; k < Dim; k++) {
                                    Type f = (*eigenVectors)[k][i+1];
                                    (*eigenVectors)[k][i+1] = s * (*eigenVectors)[k][i] + c * f;
                                    (*eigenVectors)[k][i] = c * (*eigenVectors)[k][i] - s * f;
                                }
                            }
                        }
                        if ((r == Type(0)) && (i >= (int)l))
                            continue;
                        d[l] -= p;
                        e[l] = g;
                        e[m] = Type(0);
                    }
                } while (m != l);

            }
            eigenValues = d;
        }
    private:
        Vector<Dim, Type> m_diagonal;
        Vector<Dim-1, Type> m_offDiagonal;

        inline static double copy_sign(double a, double b) {
            return copysign(a, b);
        }
        inline static float copy_sign(float a, float b) {
            return copysignf(a, b);
        }

        inline static Type pythag(const Type &a, const Type &b) {
            Type absA = std::abs(a);
            Type absB = std::abs(b);
            if (absA > absB)
                return absA * std::sqrt(Type(1)+sqr(absB/absA));
            if (absB == Type(0))
                return Type(0);
            return absB * std::sqrt(Type(1)+sqr(absA/absB));
        }
};


/// @}

}

#endif // _TRIDIAGONALSYMMETRIC_HPP_
