/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "ConfigurationReflection.h"

#include <config/SfMConfig.h>

namespace SFM {
namespace Utilities {

ConfigurationReflection::ConfigurationReflection(config::BundleAdjustmentParameterConfig &config) :
    m_root("BA-P", "Bundle adjustment parameters")
{
    m_root << &(*(new ParameterGroup("LM", "Levenberg-Marquardt parameters"))
                    << new FloatParameter("InitialLambda", "The damping parameter to start with.",
                                          &config.LevenbergMarquardt_InitialLambda)
                    << new FloatParameter("LambdaIncreaseFactor", "Factor by which the damping parameter gets increased for every misstep.",
                                          &config.LevenbergMarquardt_LambdaIncreaseFactor)
                    << new FloatParameter("LambdaDecreaseFactor", "Factor by which the damping parameter gets decreased for every successfull step.",
                                          &config.LevenbergMarquardt_LambdaDecreaseFactor)
                    << new FloatParameter("MinLambda", "The minimum bound on the damping parameter.",
                                          &config.LevenbergMarquardt_MinLambda)
                    << new FloatParameter("MaxLambdaForConvergence", "Maximal value of the damping parameter for which convergence is assumed.",
                                          &config.LevenbergMarquardt_MaxLambdaForConvergence)
                )
           << &(*(new ParameterGroup("PCGD", "Preconditioned conjugate gradient descend parameters"))
                    << new UnsignedParameter("PCGD_MaxIterations", "Maximal number of PCGD iterations per LM iteration.",
                                          &config.PCGD_MaxIterations, 1, 500)
                    << new FloatParameter("RtimesZThresholdFactor", "Threshold when to abort PCGD.",
                                          &config.PCGD_RtimesZThresholdFactor)
                );
}

namespace detail {
static const char *calibChoices[] = {"FullCalibration", "SharedFocalLength", "Fixed"};
static const char *calibChoiceDesc[] =  {"All parameters of the internal calibration may be adjusted by BA.",
                 "All parameters of the internal calibration may be adjusted by BA, but the focal lengths may only be adjusted together.",
                 "No parameters of the internal calibration may be adjusted by BA"};

static const char *radialChoices[] = {"NoRadialDistortion", "Polynomial_234"};
static const char *radialChoiceDesc[] =   {"Assume no radial distortion.",
                 "Use a polynomial model with the koefficients kappa_2 through kappa_4."};
}

ConfigurationReflection::ConfigurationReflection(config::BundleAdjustmentStructureConfig &config) :
    m_root("BA-S", "Bundle adjustment structure")
{

    m_root
        << new EnumParameterTmpl<config::BundleAdjustmentStructureConfig::InternalCalibrationType>(
                    "internalCalibrationType", "Defines to what degree the internal calibration is to be tweaked by the bundle adjustment.",
                    &config.internalCalibrationType, 3, detail::calibChoices, detail::calibChoiceDesc)

        << new EnumParameterTmpl<config::BundleAdjustmentStructureConfig::RadialDistortionType>(
                    "radialDistortionType", "Radial distortion model. For a fixed internal calibration, the distortion parameters is also fixed.",
                    &config.radialDistortionType, 2, detail::radialChoices, detail::radialChoiceDesc);

}


ConfigurationReflection::ConfigurationReflection(config::CudaConfig &config) :
    m_root("CUDA", "Cuda parameters")
{
    m_root << new StringParameter("kernelFatbinPath", "Path to the kernel fatbins", &config.kernelFatbinBaseDir);
}

ConfigurationReflection::ConfigurationReflection(config::FeatureExtractionConfig &config) :
    m_root("FE", "Feature extraction parameters")
{
    m_root
        << new FloatParameter("minimalDoGThreshold", "Threshold on the difference of gaussians strength for feature point extraction. Lower values generate more feature points.",
                                &config.minimalDoGThreshold)
        << new FloatParameter("maxElongation_rth", "Parameter that restricts how 'thin'/line like feature points can be.",
                                &config.maxElongation_rth)
        << new UnsignedParameter("maxPossibleFPLocations", "Maximum feature points per octave. Needed to reserve memory.",
                                &config.maxPossibleFPLocations)
        << new UnsignedParameter("maxFPs", "Maximum feature points per image. Needed to reserve memory.",
                                &config.maxFPs)
        << new UnsignedParameter("numOctaves", "SIFT operates on octaves. This limits the number of octaves to perform on.",
                                &config.numOctaves);
}

ConfigurationReflection::ConfigurationReflection(config::FeatureMatchingConfig &config) :
    m_root("FM", "Feature matching parameters")
{

    m_root
        << new UnsignedParameter("matchingMultithreadingBatchSize", "Number of matching operations per batch. Batches are distributed among the threads in the thread pool.",
                                &config.matchingMultithreadingBatchSize)

        << new FloatParameter("maxFeaturePointUniqueness", "Maximum value for a feature points uniqueness in case no similar feature point is found",
                                &config.maxFeaturePointUniqueness)


        << new FloatParameter("initialImagePairMatchingThreshold", "Maximal matching difference for the feature point matching of the initial image pair",
                                &config.initialImagePairMatchingThreshold)

        << new UnsignedParameter("frameGridSubdivisions", "Number of horizontal subdivisions of the grid that is used to evaluate the need to match more feature points (and also SubsetBA subset selection).",
                                &config.frameGridSubdivisions)


        << &(*(new ParameterGroup("frameNewCameraMatchingConfig", "Parameters for adding new cameras"))
            << new FloatParameter("maxNormalizedMatchDiff", "Matching threshold for the initial track vs. image feature point matching",
                                    &config.frameNewCameraMatchingConfig.maxNormalizedMatchDiff)
            << new UnsignedParameter("minInitialMatchCount", "How many matches have to be found initially for a RANSAC based estimation of the cameras pose to be attempted.",
                                    &config.frameNewCameraMatchingConfig.minInitialMatchCount)
            << new UnsignedParameter("numInitialRansacIterations", "How many RANSAC iterations to run with the initial matches.",
                                    &config.frameNewCameraMatchingConfig.numInitialRansacIterations)

            << new FloatParameter("minInitialRansacScore", "Minimal score of the best RANSAC result to continue with the boostrapping",
                                    &config.frameNewCameraMatchingConfig.minInitialRansacScore)
            << new UnsignedParameter("minBootstrappingMatchCount", "Minimal number of matches of the bootstrapping location constrained matching",
                                    &config.frameNewCameraMatchingConfig.minBootstrappingMatchCount)
            << new UnsignedParameter("numSecondaryRansacIterations", "Number of RANSAC iterations to perform on the bootstrapping matches",
                                    &config.frameNewCameraMatchingConfig.numSecondaryRansacIterations)

            << new FloatParameter("minSecondaryRansacScore", "Minimal score of the best RANSAC result from the boostrapping",
                                    &config.frameNewCameraMatchingConfig.minSecondaryRansacScore)
            << new FloatParameter("minNewObsScore", "Minimal score of each observation to be accepted",
                                    &config.frameNewCameraMatchingConfig.minNewObsScore)

            << new UnsignedParameter("minAcceptedObs", "Minimal number of accepted observations to accept the new camera/frame",
                                    &config.frameNewCameraMatchingConfig.minAcceptedObs)

        )

        << &(*(new ParameterGroup("frameBoostrappingMatchingConfig", "Parameters for the bootstrapping of a new camera (finding additional observations that link the new camera to the reconstruction)"))
            << new FloatParameter("maxSqrDistanceError", "(Soft) maximal squared distance error",
                                    &config.frameBoostrappingMatchingConfig.maxSqrDistanceError)
            << new FloatParameter("maxNormalizedMatchDiff", "(Soft) maximal matching difference",
                                    &config.frameBoostrappingMatchingConfig.maxNormalizedMatchDiff)
            << new FloatParameter("minScoreThreshold", "Minimal score of a match to be accepted",
                                    &config.frameBoostrappingMatchingConfig.minScoreThreshold)
        )
        << &(*(new ParameterGroup("frameNewObservationsMatchingConfig", "Parameters for finding new observations for already reconstructed cameras"))
            << new FloatParameter("maxSqrDistanceError", "(Soft) maximal squared distance error",
                                    &config.frameNewObservationsMatchingConfig.maxSqrDistanceError)
            << new FloatParameter("maxNormalizedMatchDiff", "(Soft) maximal matching difference",
                                    &config.frameNewObservationsMatchingConfig.maxNormalizedMatchDiff)
            << new FloatParameter("minScoreThreshold", "Minimal score of a match to be accepted",
                                    &config.frameNewObservationsMatchingConfig.minScoreThreshold)
        )
        << &(*(new ParameterGroup("frameNewTracksMatchingConfig", "Parameters for finding new tracks"))
            << new FloatParameter("maxDistanceError", "(Soft) maximal distance error",
                                    &config.frameNewTracksMatchingConfig.maxDistanceError)
            << new FloatParameter("maxNormalizedMatchDiff", "(Soft) maximal matching difference",
                                    &config.frameNewTracksMatchingConfig.maxNormalizedMatchDiff)

            << new UnsignedParameter("maxNumActiveObsPerCell", "Maximal number of active observations per cell when to stop looking for new tracks for that cell",
                                    &config.frameNewTracksMatchingConfig.maxNumActiveObsPerCell)
            << new FloatParameter("minObservationUniqueness", "Minimal uniqueness of an observation to use it for matching",
                                    &config.frameNewTracksMatchingConfig.minObservationUniqueness)
            << new UnsignedParameter("maxObservationsTestedPerCellNum", "Maximal number of observations to use for finding new tracks per trial and cell",
                                    &config.frameNewTracksMatchingConfig.maxObservationsTestedPerCellNum)

            << new FloatParameter("minScoreThreshold", "Minimal score of a match to be accepted",
                                    &config.frameNewTracksMatchingConfig.minScoreThreshold)
        );
}

ConfigurationReflection::ConfigurationReflection(config::SubsetSelectionConfig &config) :
    m_root("SubSel", "Bundle adjustment subset selection parameters")
{
    m_root
        << new UnsignedParameter("numCells", "How many grid cells the regions span in each direction",
                                &config.numCells)
        << new UnsignedParameter("numVotesPerRegion", "How many votes a frame gets per region",
                                &config.numVotesPerRegion)

        << new FloatParameter("scoreSqrScreenSpaceErrorFactor", "How strongly to penalize reprojection errors when selecting the subset",
                                &config.scoreSqrScreenSpaceErrorFactor);
}

ConfigurationReflection::ConfigurationReflection(config::SfMConfig &config) :
    m_root("SfM", "Overall SfM parameters")
{

    m_root
        << new FloatParameter("outlierRemovalActiveThreshFactor", "Factor on the avg. reprojection error to determine the reprojection error threshold for identifying outliers among the active tracks",
                                &config.outlierRemovalActiveThreshFactor)
        << new FloatParameter("outlierRemovalActiveThreshMinimum", "Minimal reprojection error threshold for identifying outliers among the active tracks",
                                &config.outlierRemovalActiveThreshMinimum)

        << new FloatParameter("outlierRemovalSemiActiveThreshFactor", "Factor on the avg. reprojection error to determine the reprojection error threshold for identifying outliers among the semi-active tracks",
                                &config.outlierRemovalSemiActiveThreshFactor)
        << new FloatParameter("outlierRemovalSemiActiveThreshMinimum", "Minimal reprojection error threshold for identifying outliers among the semi-active tracks",
                                &config.outlierRemovalSemiActiveThreshMinimum)

        << new UnsignedParameter("baSemiActiveTracksBatchSize", "Batch size for multithreaded structure-only bundle adjustment of the semi-active tracks",
                                &config.baSemiActiveTracksBatchSize)
        << new UnsignedParameter("numRemoveOutlierItertions", "Number of BundleAdjustment <--> OutlierRemoval iterations",
                                &config.numRemoveOutlierItertions)
        << new UnsignedParameter("numExtraRounds", "Number of extra rounds to perform after the last camera was added to the reconstruction",
                                &config.numExtraRounds)
        << new UnsignedParameter("maxRoundsWithoutNewImage", "Maximal number of rounds without adding a new camera before the reconstruction is aborted",
                                &config.maxRoundsWithoutNewImage);
}


}
}
