/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "DotGenerator.h"

#include <SFM/SFM.h>
#include <SFM/Track.h>
#include <SFM/Frame.h>
#include <SFM/InternalCameraCalibration.h>


#include <fstream>
#include <map>


namespace SFM {
namespace Utilities {

void generateDotFile(const boost::filesystem::path &destinationPath, SFM &sfm, bool onlySubset, bool fullFilenames)
{
    std::map<std::pair<unsigned, unsigned>, float> edgeMap;

    for (const Track &track : sfm.getTrackList()) {
        if (track.getState() == Track::STATE_DISABLED) continue;
        if (onlySubset && (track.getState() == Track::STATE_SEMI_ACTIVE)) continue;


        for (unsigned k = 0; k < track.getObservations().reservedSize(); k++) {
            if (!track.getObservations().inUse(k)) continue;
            const TrackObservation &obs1 = track.getObservations()[k];
            if (obs1.isFaulty())
                continue;

            unsigned obs1FrameIndex = obs1.getFrame()->getIndex();

            for (unsigned l = k+1; l < track.getObservations().reservedSize(); l++) {
                if (!track.getObservations().inUse(l)) continue;
                const TrackObservation &obs2 = track.getObservations()[l];
                if (obs2.isFaulty())
                    continue;

                unsigned obs2FrameIndex = obs2.getFrame()->getIndex();

                //assert(obs1FrameIndex != obs2FrameIndex);

                if (obs1FrameIndex == obs2FrameIndex) continue;

                auto key = (obs1FrameIndex < obs2FrameIndex)?
                                std::pair<unsigned, unsigned>(obs1FrameIndex, obs2FrameIndex):
                                std::pair<unsigned, unsigned>(obs2FrameIndex, obs1FrameIndex);

                edgeMap[key]+=1.0f;
            }
        }
    }

    std::fstream file;
    file.exceptions(std::ifstream::failbit | std::ifstream::badbit);
    file.open(destinationPath.native().c_str(), std::fstream::out);

    file << "graph cameragraph {" << std::endl;

    for (unsigned i = 0; i < sfm.getFrames().size(); i++) {
        if (fullFilenames)
            file << "    cam"<<i<<" [label=\""<<sfm.getFrames()[i]->getImageFilename()<<"\"];" << std::endl;
        else
            file << "    cam"<<i<< std::endl;
    }

    for (const auto &edge : edgeMap) {
        file << "    cam"<<edge.first.first << " -- cam" << edge.first.second << " [len="<<1000.0f / edge.second<<"];"<<std::endl;
    }


    file << "}" << std::endl;
}

}
}

