/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/**


@page Page_CommandLineFrontend The Command Line Frontend

@section Page_CommandLineFrontend_Intro Introduction

The command line frontend is a simple command line program
to run sparse reconstructions with the implemented
SFM backend. Input images, parameters, and output filenames
are specified via command line arguments to the program.

@section Page_CommandLineFrontend_Usage Usage

@subsection Page_CommandLineFrontend_Usage_Example Example
The following demonstrates a sparse reconstruction
followed by a dense reconstruction with PMVS2. Using primusrun (or optirun)
is only needed when the GPU is subject to NVidia Optimus (energy saving mechanism on notebooks).
@code{.unparsed}
$ primusrun path/to/CommandLineFrontend --initial-internal-calib-FocalLength 31 --save-pmvs ./HerzJesuP25 --initial-pair 7 8 --filenames ../data/Herz-Jesu-P25/*.jpg
$ pmvs2 ./HerzJesuP25 ./HerzJesuP25/options.txt
@endcode

The tool can also generate "binSfm" files, which can be inspected with the @ref Page_SparseReconstructionViewer "sparse reconstruction viewer".

@subsection Page_CommandLineFrontend_Usage_Important_Parameters Important Parameters

Running the program without any arguments will give an explanation of the most important parameters.
The parameter "--filenames" is mandatory and specifies a list of images, relative to the current working directory. All
arguments following "--filenames" are interpreted as filenames which is why the list of filenames must be specified last.
The parameter "--initial-pair" is also mandatory and specifies the (zero-based) indices of the image pair to start with.
Specifying the initial internal calibration is also mandatory but can be performed by one of three methods. The
argument "--initial-internal-calib-FocalLength" is probably the easiest, as it initializes the internal calibration
to a neutral state (square pixels, no shear, principal point is the image center) and allows the specification of
the focal length in 35mm film equivalent. For the remaining two commands see "path/to/CommandLineFrontend --help".
Note that actually saving the results of the reconstruction is not mandatory. Several output options exist.
The argument "--save-pmvs" is one of them and prepares the directories, data, and configuration files for a subsequent
PMVS2 based dense reconstruction. For other output options see "path/to/CommandLineFrontend --help".

@subsubsection Page_CommandLineFrontend_Usage_Important_Parameters_focalLength A Note on Focal Lengths 
Note that the value that most cameras store in the jpeg metadata fields is not
corrected for the cameras' crop factors. For example, in a not-full-frame Canon camera, the image sensor's size
is reduced by a factor of 1.6 (the crop factor). The jpeg files, however, still store the focal length as reported
by the lens (in 35mm equivalents). Due to the reduced sensor size, the effective focal length is actually 1.6 times larger.
This has to be accounted for when specifying the focal length for the SfM pipeline. So, when you are shooting with a 20mm lens
on a camera with a crop factor of 1.6, you should specify "--initial-internal-calib-FocalLength 32" (20mm * 1.6 = 32mm) for
the internal calibration.

 */





#include <iostream>

#include <cudaUtilities/CudaDriver.h>
#include <cudaUtilities/CudaDevice.h>
#include <cudaUtilities/CudaDeviceContext.h>

#include <tools/TaskScheduler.h>

#include <SFM/SFM.h>

#include <ConfigurationReflection.h>

#include <config/SfMConfig.h>

#include <PMVSExporter.h>
#include <XmlExporter.h>
#include <BinSFMExporter.h>
#include <BundlerExporter.h>
#include <DotGenerator.h>
#include <MinimumCutAnalysis.h>

#include <boost/algorithm/string/predicate.hpp>


#include <SFM/Track.h>
#include <SFM/TrackObservation.h>
#include <SFM/Frame.h>
#include <SFM/InternalCameraCalibration.h>


void ReccurPrintOptions(SFM::Utilities::ConfigurationReflection::ParameterGroup &group, const std::string &indentation)
{
    std::cout << indentation << "+- " << group.getName() << std::endl;
    std::cout << indentation << "|    |    " << group.getDescription() << std::endl;

    std::string indent = indentation + "|    ";
    for (SFM::Utilities::ConfigurationReflection::BaseElement &element : group) {

        if (typeid(element) == typeid(SFM::Utilities::ConfigurationReflection::ParameterGroup)) {
            ReccurPrintOptions(static_cast<SFM::Utilities::ConfigurationReflection::ParameterGroup &>(element), indent);
        } else {
            SFM::Utilities::ConfigurationReflection::BaseParameter &baseParam =
                    static_cast<SFM::Utilities::ConfigurationReflection::BaseParameter&>(element);

            std::cout << indent << "+- " << baseParam.getName() << std::endl;
            std::cout << indent << "|     " << baseParam.getDescription() << std::endl;
            std::cout << indent << "|     Default value: " << baseParam.getValueAsStr() << std::endl;

            SFM::Utilities::ConfigurationReflection::NumericParameter *numericParam =
                    dynamic_cast<SFM::Utilities::ConfigurationReflection::NumericParameter*>(&element);
            if (numericParam != NULL) {
                std::cout << indent << "|     Valid range: " << numericParam->getMinValueAsStr() << " .. " << numericParam->getMaxValueAsStr() << std::endl;
            }

            SFM::Utilities::ConfigurationReflection::BaseEnumParameter *enumParam =
                    dynamic_cast<SFM::Utilities::ConfigurationReflection::BaseEnumParameter*>(&element);
            if (enumParam != NULL) {
                std::cout << indent << "|     Valid choices: " << std::endl;
                for (unsigned i = 0; i < enumParam->getNumChoices(); i++) {
                    std::cout << indent << "|         " << enumParam->getChoiceName(i) << std::endl;
                    std::cout << indent << "|             " << enumParam->getChoiceDescription(i) << std::endl;
                }
            }
        }
    }
    std::cout << indentation << '|' << std::endl;
}

void displayHelp()
{
    std::cout << "Usage:" << std::endl;
    std::cout << "CommandLineFrontend --help" << std::endl
              << "    lists this help" << std::endl;
    std::cout << "CommandLineFrontend [options] <initialInternalCalibration> --initial-pair <index1> <index2> --filenames <filenames>" << std::endl
              << "    Runs SfM on the images listed as <filenames>. <index1> and <index2> denote the initial image pair to start with."<< std::endl;

    std::cout << "    <initialInternalCalibration> must be specified by one of the following methods:" << std::endl
              << "         --initial-internal-calib-FOV <fieldOfViewInDeg>" << std::endl
              << "                Sets the focal length so that the cameras have specified horizontal field of view. Otherwise no shear, square pixels and principal point in the image center are assumed." << std::endl
              << "         --initial-internal-calib-FocalLength <focalLength>" << std::endl
              << "                Uses the 35mm film equivalent of <focalLength> as the focal lengths. Otherwise no shear, square pixels and principal point in the image center are assumed." << std::endl
              << "         --initial-internal-calib-Full <fx> <fy> <px> <py> <s>" << std::endl
              << "                Sets all five parameters. Horizontal focal length (fx), vertical focal length (fy), principal point (px, py), and shear (s). The coordinate system is centered in the image center. The left and right image borders are at x=+-1 so a focal length of 1 equals a horizontal field of view of 90 deg." << std::endl;

    std::cout << "    [options] may be any of the following:" << std::endl;
    std::cout << "        --print-calib                       print the internal calibration of all cameras" << std::endl;
    std::cout << "        --save-xml <filename>               save the results into a xml file" << std::endl;
    std::cout << "        --save-binsfm <filename>            save the results into a binsfm file" << std::endl;
    std::cout << "        --save-bundler <filename>           save the results into a bundler file" << std::endl;
    std::cout << "        --save-pmvs <directory>             prepares the directory structure and files needed to run pmvs2" << std::endl;
    std::cout << "        --generate-dot-full <filename>      generates a dot file of the full camera graph (for tools like Graphviz)" << std::endl;
    std::cout << "        --generate-dot-subset <filename>    generates a dot file of the subset camera graph (for tools like Graphviz)" << std::endl;
    std::cout << "        --analyze-mincut-full               computes the minimum cut of the camera graph" << std::endl;
    std::cout << "        --analyze-mincut-subset             computes the minimum cut of the camera graph, but only considers the subset used for bundle adjustment" << std::endl;
    std::cout << "        or any of the parameters specified by --list-parameters. These can be specified like this:" << std::endl;
    std::cout << "            --param-BA-S-radialDistortionType NoRadialDistortion" << std::endl;
}



SFM::Utilities::ConfigurationReflection::BaseParameter *reccurTryMatchReflectedConfig(SFM::Utilities::ConfigurationReflection::ParameterGroup &group,
                                                                                      const std::string &prefix, const std::string &name)
{
    std::string paramName = prefix+'-'+group.getName();

    if (!boost::starts_with(name, paramName))
        return nullptr;

    for (SFM::Utilities::ConfigurationReflection::BaseElement &element : group) {

        if (typeid(element) == typeid(SFM::Utilities::ConfigurationReflection::ParameterGroup)) {
            SFM::Utilities::ConfigurationReflection::BaseParameter *param = reccurTryMatchReflectedConfig(
                    static_cast<SFM::Utilities::ConfigurationReflection::ParameterGroup &>(element), paramName, name);

            if (param != nullptr)
                return param;
        } else {
            SFM::Utilities::ConfigurationReflection::BaseParameter &baseParam =
                    static_cast<SFM::Utilities::ConfigurationReflection::BaseParameter&>(element);

            if (name == (paramName+'-'+baseParam.getName()))
                return &baseParam;
        }
    }
    return nullptr;
}


SFM::Utilities::ConfigurationReflection::BaseParameter *tryMatchReflectedConfig(SFM::Utilities::ConfigurationReflection &reflectionConfig, const std::string &name)
{
    return reccurTryMatchReflectedConfig(reflectionConfig.getReflection(), "--param", name);
}


int main(int argc, const char **argv)
{
    try {
        std::cout << "Command line SFM frontend" << std::endl;
#ifndef __AVX__
        std::cout << "WARNING: This executable was compiled without AVX support, so it's falling back to a slower backup codepath!" << std::endl;
#endif        
        if (argc == 1) {
            displayHelp();
            return 0;
        }

        SFM::config::SfMConfig sfmConfig;

        SFM::Utilities::ConfigurationReflection baParameterConfigReflection(sfmConfig.bundleAdjustmentParameterConfig);
        SFM::Utilities::ConfigurationReflection baStructureConfigReflection(sfmConfig.bundleAdjustmentStructureConfig);
        SFM::Utilities::ConfigurationReflection cudaConfigReflection(sfmConfig.cudaConfig);
        SFM::Utilities::ConfigurationReflection featureExtractionConfigReflection(sfmConfig.featureExtractionConfig);
        SFM::Utilities::ConfigurationReflection featureMatchingConfigReflection(sfmConfig.featureMatchingConfig);
        SFM::Utilities::ConfigurationReflection subsetSelectionConfigReflection(sfmConfig.subsetSelectionConfig);
        SFM::Utilities::ConfigurationReflection SfMConfigReflection(sfmConfig);

        boost::optional<std::string> saveXML;
        boost::optional<std::string> saveBinsfm;
        boost::optional<std::string> saveBundler;
        boost::optional<std::string> savePMVS;
        boost::optional<std::string> generateDotSubset;
        boost::optional<std::string> generateDotFull;

        bool analyzeMinCutFull = false;
        bool analyzeMinCutSubset = false;

        boost::optional<LinAlg::Matrix4x4f> P;
        boost::optional<unsigned> initialImagePairIndex1;
        boost::optional<unsigned> initialImagePairIndex2;

        bool printCalibrationEstimation = false;

        std::vector<std::string> filenames;

        SFM::Utilities::ConfigurationReflection::BaseParameter *baseParameter;

        for (int argi = 1; argi < argc; argi++) {
            std::string arg(argv[argi]);

            if (arg == "--list-parameters") {
                ReccurPrintOptions(baParameterConfigReflection.getReflection(),"");
                ReccurPrintOptions(baStructureConfigReflection.getReflection(),"");
                ReccurPrintOptions(cudaConfigReflection.getReflection(),"");
                ReccurPrintOptions(featureExtractionConfigReflection.getReflection(),"");
                ReccurPrintOptions(featureMatchingConfigReflection.getReflection(),"");
                ReccurPrintOptions(subsetSelectionConfigReflection.getReflection(),"");
                ReccurPrintOptions(SfMConfigReflection.getReflection(),"");
                return 0;
            } else
            if (arg == "--help") {
                displayHelp();
                return 0;
            } else
            if (arg == "--print-calib") {
                printCalibrationEstimation = true;
            } else
            if (arg == "--save-xml") {
                if (argi+1 == argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                saveXML = std::string(argv[++argi]);
            } else
            if (arg == "--save-bundler") {
                if (argi+1 == argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                saveBundler = std::string(argv[++argi]);
            } else
            if (arg == "--save-pmvs") {
                if (argi+1 == argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                savePMVS = std::string(argv[++argi]);
            } else
            if (arg == "--save-binsfm") {
                if (argi+1 == argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                saveBinsfm = std::string(argv[++argi]);
            } else
            if (arg == "--generate-dot-full") {
                if (argi+1 == argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                generateDotFull = std::string(argv[++argi]);
            } else
            if (arg == "--generate-dot-subset") {
                if (argi+1 == argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                generateDotSubset = std::string(argv[++argi]);
            } else
            if (arg == "--analyze-mincut-full") {
                analyzeMinCutFull = true;
            } else
            if (arg == "--analyze-mincut-subset") {
                analyzeMinCutSubset = true;
            } else
            if (arg == "--initial-internal-calib-FOV") {
                if (argi+1 >= argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                try {
                    LinAlg::Matrix4x4f P_;

                    float fov = boost::lexical_cast<float>(argv[++argi]) *(float)M_PI/180.0f;

                    P_[0][0] =
                    P_[1][1] = 1.0f / tan(0.5f*fov);

                    P_[2][0] = 0.0f;
                    P_[2][1] = 0.0f;
                    P_[2][2] = 0.0f;
                    P_[2][3] = 1.0f;

                    P_[3][0] = 0.0f;
                    P_[3][1] = 0.0f;
                    P_[3][2] = 1.0f;
                    P_[3][3] = 0.0f;

                    P = P_;
                } catch (const boost::bad_lexical_cast &) {
                    std::cerr << "Invalid floating point value while parsing " << arg << " arguments" << std::endl;
                    return -1;
                }
            } else
            if (arg == "--initial-internal-calib-Full") {
                if (argi+5 >= argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                try {
                    LinAlg::Matrix4x4f P_;

                    float fx = boost::lexical_cast<float>(argv[++argi]);
                    float fy = boost::lexical_cast<float>(argv[++argi]);
                    float px = boost::lexical_cast<float>(argv[++argi]);
                    float py = boost::lexical_cast<float>(argv[++argi]);
                    float s = boost::lexical_cast<float>(argv[++argi]);

                    P_[0][0] = fx;
                    P_[0][1] = s;
                    P_[0][2] = px;

                    P_[1][1] = fy;
                    P_[1][2] = py;

                    P_[2][0] = 0.0f;
                    P_[2][1] = 0.0f;
                    P_[2][2] = 0.0f;
                    P_[2][3] = 1.0f;

                    P_[3][0] = 0.0f;
                    P_[3][1] = 0.0f;
                    P_[3][2] = 1.0f;
                    P_[3][3] = 0.0f;

                    P = P_;
                } catch (const boost::bad_lexical_cast &) {
                    std::cerr << "Invalid floating point value while parsing " << arg << " arguments" << std::endl;
                    return -1;
                }
            } else
            if (arg == "--initial-internal-calib-FocalLength") {
                if (argi+1 >= argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                try {
                    LinAlg::Matrix4x4f P_;

                    float f = boost::lexical_cast<float>(argv[++argi]) / (36.0f * 0.5f);

                    P_[0][0] =
                    P_[1][1] = f;

                    P_[2][0] = 0.0f;
                    P_[2][1] = 0.0f;
                    P_[2][2] = 0.0f;
                    P_[2][3] = 1.0f;

                    P_[3][0] = 0.0f;
                    P_[3][1] = 0.0f;
                    P_[3][2] = 1.0f;
                    P_[3][3] = 0.0f;

                    P = P_;
                } catch (const boost::bad_lexical_cast &) {
                    std::cerr << "Invalid floating point value while parsing " << arg << " arguments" << std::endl;
                    return -1;
                }
            } else
            if (arg == "--initial-pair") {
                if (argi+2 >= argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                try {
                    initialImagePairIndex1 = boost::lexical_cast<unsigned>(argv[++argi]);
                    initialImagePairIndex2 = boost::lexical_cast<unsigned>(argv[++argi]);
                } catch (const boost::bad_lexical_cast &) {
                    std::cerr << "Invalid integer value while parsing " << arg << " arguments" << std::endl;
                    return -1;
                }
            } else
            if (arg == "--filenames") {
                for (argi++; argi < argc; argi++) {
                    filenames.push_back(argv[argi]);
                }
            } else
            if ((baseParameter = tryMatchReflectedConfig(baStructureConfigReflection, arg)) != nullptr) {
                if (argi+1 >= argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                if (!baseParameter->parseValueFromString(argv[++argi])) {
                    std::cerr << "Invalid parameter to " << arg << std::endl;
                    return -1;
                }
            } else
            if ((baseParameter = tryMatchReflectedConfig(baParameterConfigReflection, arg)) != nullptr) {
                if (argi+1 >= argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                if (!baseParameter->parseValueFromString(argv[++argi])) {
                    std::cerr << "Invalid parameter to " << arg << std::endl;
                    return -1;
                }
            } else
            if ((baseParameter = tryMatchReflectedConfig(cudaConfigReflection, arg)) != nullptr) {
                if (argi+1 >= argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                if (!baseParameter->parseValueFromString(argv[++argi])) {
                    std::cerr << "Invalid parameter to " << arg << std::endl;
                    return -1;
                }
            } else
            if ((baseParameter = tryMatchReflectedConfig(featureExtractionConfigReflection, arg)) != nullptr) {
                if (argi+1 >= argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                if (!baseParameter->parseValueFromString(argv[++argi])) {
                    std::cerr << "Invalid parameter to " << arg << std::endl;
                    return -1;
                }
            } else
            if ((baseParameter = tryMatchReflectedConfig(featureMatchingConfigReflection, arg)) != nullptr) {
                if (argi+1 >= argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                if (!baseParameter->parseValueFromString(argv[++argi])) {
                    std::cerr << "Invalid parameter to " << arg << std::endl;
                    return -1;
                }
            } else
            if ((baseParameter = tryMatchReflectedConfig(subsetSelectionConfigReflection, arg)) != nullptr) {
                if (argi+1 >= argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                if (!baseParameter->parseValueFromString(argv[++argi])) {
                    std::cerr << "Invalid parameter to " << arg << std::endl;
                    return -1;
                }
            } else
            if ((baseParameter = tryMatchReflectedConfig(SfMConfigReflection, arg)) != nullptr) {
                if (argi+1 >= argc) {
                    std::cerr << "Insufficient arguments" << std::endl;
                    return -1;
                }
                if (!baseParameter->parseValueFromString(argv[++argi])) {
                    std::cerr << "Invalid parameter to " << arg << std::endl;
                    return -1;
                }
            } else {
                std::cerr << "Unknown argument: " << arg << std::endl;
                return -1;
            }
        }

        if (!P) {
            std::cerr << "No initial projection matrix was supplied!" << std::endl;
            return -1;
        }

        if ((!initialImagePairIndex1) || (!initialImagePairIndex2)) {
            std::cerr << "No initial image pair was supplied!" << std::endl;
            return -1;
        }

        if ((*initialImagePairIndex1 >= filenames.size()) || (*initialImagePairIndex2 >= filenames.size())) {
            std::cerr << "Initial image pair indices bigger than number of images!" << std::endl;
            return -1;
        }

        std::cout << "Using " << boost::thread::hardware_concurrency() << " helper threads" << std::endl;
        TaskScheduler::Init(boost::thread::hardware_concurrency());

        CudaUtils::CudaDriver driver;
        std::unique_ptr<CudaUtils::CudaDevice> device(driver.getMaxCCDevice());

        if (device == NULL) {
            std::cout << "No compute device found!" << std::endl;
            return -1;
        }

        std::cout << "Using device " << device->getDeviceName() << std::endl;
        std::unique_ptr<CudaUtils::CudaDeviceContext> context(new CudaUtils::CudaDeviceContext(device.get()));
        context->setBankSize4Byte();
        context->setPreferredSharedMem();
        context->makeCurrent();
        std::cout << "Cuda context set up" << std::endl;


        SFM::BundleAdjustment::RadialDistortionParametrization radialDistortion;
        memset(&radialDistortion, 0, sizeof(SFM::BundleAdjustment::RadialDistortionParametrization));
        radialDistortion.type = sfmConfig.bundleAdjustmentStructureConfig.radialDistortionType;


        float aspectRatio;
        { /// todo This is not very nice :-(
            RasterImage dummy;
            dummy.loadFromFile(filenames[0].c_str());
            aspectRatio = dummy.getHeight() / (float) dummy.getWidth();
        }

//sfmConfig.verbatimLevel = 1000;
        SFM::SFM sfm(sfmConfig);
        sfm.initSingleCameraPath(*P, radialDistortion, aspectRatio, filenames);

        std::cout << "Extracting patches..." << std::endl;
        sfm.extractFeaturePoints([](const SFM::SFM::ExtractFeaturePointsProgressReport &progressReport) {
            std::cout << "Processing image " << progressReport.imageFilename
                    << " (" << progressReport.imageNumber<<'/'<<progressReport.totalImageNumber << ")" << std::endl;
            std::cout << "    avg " << progressReport.numfeaturePointsSoFar/(progressReport.imageNumber+1) << " fpts/image" << std::endl;
        });


        std::cout << "Building initial pose" << std::endl;
        sfm.buildInitialPose(*initialImagePairIndex1, *initialImagePairIndex2);


        std::cout << "Running reconstruction" << std::endl;
        try {
            while (sfm.getState() != SFM::SFM::OP_STATE_DONE) {
                sfm.operate();
                if (sfm.getState() == SFM::SFM::OP_STATE_ADDPATCHES) {
                    unsigned numActiveFrames = 0;
                    for (unsigned i = 0; i < sfm.getFrames().size(); i++)
                        if (sfm.getFrames()[i]->active())
                            numActiveFrames++;

                    std::cout << "  " << numActiveFrames << " of " << sfm.getFrames().size() << " reconstructed" << std::endl;
                }
            }
        } catch (const std::exception &e) {
            std::cerr << "Something went wrong: " << e.what() << std::endl;
            std::cerr << "Trying to salvage results" << std::endl;
        }
        std::cout << "Processing results" << std::endl;

        if (printCalibrationEstimation) {
            for (const auto &calib : sfm.getInternalCameraCalibrations()) {
                std::cout << "== Internal Calibration ==" << std::endl;
                std::cout << "Projection:" << std::endl;
                std::cout << (std::string) calib->getProjectionMatrix()[0] << std::endl;
                std::cout << (std::string) calib->getProjectionMatrix()[1] << std::endl;
                std::cout << (std::string) calib->getProjectionMatrix()[2] << std::endl;
                std::cout << (std::string) calib->getProjectionMatrix()[3] << std::endl;
                std::cout << "Focal length X (35mm equiv): " << calib->getProjectionMatrix()[0][0] / calib->getProjectionMatrix()[3][2] * 36.0f*0.5f << "mm" << std::endl;
                std::cout << "Focal length Y (35mm equiv): " << calib->getProjectionMatrix()[1][1] / calib->getProjectionMatrix()[3][2] * 36.0f*0.5f << "mm" << std::endl;

                switch (calib->getRadialDistortion().type) {
                    case SFM::config::BundleAdjustmentStructureConfig::RadialDistortionType::NoRadialDistortion:
                        std::cout << "No radial distortion!" << std::endl;
                    break;
                    case SFM::config::BundleAdjustmentStructureConfig::RadialDistortionType::Polynomial_234:
                        std::cout << "Radial distortion: Polynomial_234 - kappa_{234} = "
                                << calib->getRadialDistortion().polynomial234.kappa[0] << " "
                                << calib->getRadialDistortion().polynomial234.kappa[1] << " "
                                << calib->getRadialDistortion().polynomial234.kappa[2] << std::endl;
                    break;
                }
            }
        }

        if (analyzeMinCutFull) {
            SFM::Utilities::CameraGraphMinimumCut minCut = SFM::Utilities::analyzeMinCut(sfm, false);
            std::cout << "The minimum cut capacity of the full graph is " <<
                    minCut.minimumCutCapacity << " of a total capacity of " << minCut.totalCapacity << std::endl;
            std::cout << "     " << minCut.minimumCutCapacity*100.0f/minCut.totalCapacity << " %" << std::endl;
            std::cout << "One set of vertices consists of: ";
            for (unsigned i : minCut.set1) {
                std::cout << " " << i;
            }
            std::cout << std::endl;
            std::cout << "The other set of vertices consists of: ";
            for (unsigned i : minCut.set2) {
                std::cout << " " << i;
            }
            std::cout << std::endl;
        }
        if (analyzeMinCutSubset) {
            SFM::Utilities::CameraGraphMinimumCut minCut = SFM::Utilities::analyzeMinCut(sfm, true);
            std::cout << "The minimum cut capacity of the subset graph is " <<
                    minCut.minimumCutCapacity << " of a total capacity of " << minCut.totalCapacity << std::endl;
            std::cout << "     " << minCut.minimumCutCapacity*100.0f/minCut.totalCapacity << " %" << std::endl;
            std::cout << "One set of vertices consists of: ";
            for (unsigned i : minCut.set1) {
                std::cout << " " << i;
            }
            std::cout << std::endl;
            std::cout << "The other set of vertices consists of: ";
            for (unsigned i : minCut.set2) {
                std::cout << " " << i;
            }
            std::cout << std::endl;
        }

        if (generateDotFull) {
            boost::filesystem::path path(*generateDotFull);
            std::cout << "Generating dot file: " << path.native() << std::endl;
            SFM::Utilities::generateDotFile(path, sfm, false, true);
        }
        if (generateDotSubset) {
            boost::filesystem::path path(*generateDotSubset);
            std::cout << "Generating dot file: " << path.native() << std::endl;
            SFM::Utilities::generateDotFile(path, sfm, true, true);
        }

        if (savePMVS) {
            boost::filesystem::path path(*savePMVS);
            std::cout << "Generating pmvs files: " << path.native() << std::endl;
            SFM::Utilities::exportPMVS(path, sfm);
        }

        if (saveXML) {
            boost::filesystem::path path(*saveXML);
            std::cout << "Generating xml file: " << path.native() << std::endl;
            SFM::Utilities::exportXml(path, sfm);
        }

        if (saveBundler) {
            boost::filesystem::path path(*saveBundler);
            std::cout << "Generating bundler file: " << path.native() << std::endl;
            SFM::Utilities::exportBundler(path, sfm);
        }

        if (saveBinsfm) {
            boost::filesystem::path path(*saveBinsfm);
            std::cout << "Generating binsfm file: " << path.native() << std::endl;
            SFM::Utilities::exportBinSFM(path, sfm);
        }

        std::cout << "All done. Exiting..." << std::endl;

        return 0;
    } catch (const std::exception &e) {
        std::cerr << "An unhandled exception was encountered: " << e.what() << std::endl;
        return -1;
    }
}
