/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "VertexArrayObject.h"

#include "RenderDevice.h"
#include "BufferObject.h"
#include "MeshBufferLayout.h"

#include <GL/glew.h>

namespace Engine {
namespace Graphics {


VertexArrayObject::VertexArrayObject(RenderDevice *renderDevice) : m_renderDevice(renderDevice)
{
    glGenVertexArrays(1, &m_objectId);
    m_attachedAttribsIndices = 0;
}

VertexArrayObject::~VertexArrayObject()
{
    glDeleteVertexArrays(1, &m_objectId);
}

void VertexArrayObject::attachAttribute(unsigned attribIndex, BufferObject *bufferObject, const MeshBufferLayout &bufferLayout, unsigned bufferIndex)
{
    m_renderDevice->bindVertexArrayObject(this);
    glEnableVertexAttribArray(attribIndex);
    glBindBuffer(GL_ARRAY_BUFFER, bufferObject->getID());
    glVertexAttribPointer(attribIndex,
                          bufferLayout.attribs[bufferIndex].dataType.numComponents,
                          bufferLayout.attribs[bufferIndex].dataType.dataType,
                          bufferLayout.attribs[bufferIndex].dataType.normalized?GL_TRUE:GL_FALSE,
                          bufferLayout.stride,
                          (void*) (size_t) bufferLayout.attribs[bufferIndex].offset);

    m_attachedAttribsIndices |= 1 << attribIndex;
}

void VertexArrayObject::removeAttribute(unsigned attribIndex)
{
    m_renderDevice->bindVertexArrayObject(this);
    glDisableVertexAttribArray(attribIndex);

    m_attachedAttribsIndices &= ~(1 << attribIndex);
}

void VertexArrayObject::removeAllAttributes()
{
    m_renderDevice->bindVertexArrayObject(this);
    for (unsigned i = 0; i < 32; i++) {
        if ((m_attachedAttribsIndices >> i) == 0)
            break;

        if (m_attachedAttribsIndices & (1 << i))
            glDisableVertexAttribArray(i);
    }

    m_attachedAttribsIndices = 0;
}


void VertexArrayObject::attachIndexArray(BufferObject *indexBufferObject)
{
    m_renderDevice->bindVertexArrayObject(this);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, indexBufferObject->getID());
}


}
}
