/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef BASEWINDOW_H
#define BASEWINDOW_H

#include <vector>
#include <string>

namespace Engine {
namespace Platform {


class RenderContextInterface
{
    public:
        virtual ~RenderContextInterface();

        virtual void makeCurrent() = 0;
        virtual void swap() = 0;
    private:
};


class BaseWindow
{
    public:
        struct WindowParams {
            std::string windowName;
            unsigned width;
            unsigned height;
            bool fullscreen;
            bool grabInput;
            std::vector<int> contextAttributes;
            std::vector<int> visualAttributes;
        };

        union Event {
            enum EventType {
                TYPE_MOTION,
                TYPE_BUTTON_DOWN,
                TYPE_BUTTON_UP,
                TYPE_KEY_DOWN,
                TYPE_KEY_UP,
                TYPE_NOTIFICATION
            };
            EventType type;

            struct Motion {
                EventType type;
                int x, y;
                float relX, relY;
            };
            struct Button {
                EventType type;
                int x, y;
                unsigned index;
            };
            struct Key {
                EventType type;
                int x, y;
                unsigned keysym;
            };
            struct Notification {
                EventType type;
                enum NotificationMessage {
                    NM_WMSHUTDOWN,
                    NM_RESIZE
                };
                NotificationMessage message;
            };

            Motion motion;
            Button button;
            Key key;
            Notification notification;
        };

        struct State {
            int mouseX;
            int mouseY;
            float mouseRelX;
            float mouseRelY;

            unsigned keyState[32];
            unsigned buttonState;
            bool shutdown;

            State();
        };


        BaseWindow();
        virtual ~BaseWindow();

        virtual void resetWindow(WindowParams &params) = 0;

        virtual bool getNextEvent(Event &event) = 0;
        inline const State &getState() const { return m_state; }

        inline RenderContextInterface *getContext() { return m_renderContext; }

        inline const WindowParams &getParams() const { return m_params; }
    protected:
        RenderContextInterface *m_renderContext;
        WindowParams m_params;
        State m_state;
};

}
}

#endif // BASEWINDOW_H
