/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef BIFOCALRANSACFILTER_H
#define BIFOCALRANSACFILTER_H

#include "../tools/LinAlg.h"
#include <immintrin.h>


namespace SFM {

/**
 * @brief RANSAC based filter which finds a fundamental matrix from a set of matches between two images.
 * @ingroup SFMBackend_Group
 */
class BifocalRANSACFilter
{
    public:
        struct Match {
            unsigned index[2];
            LinAlg::Vector2f pos[2] __attribute__((aligned(16)));
            float size[2];
            float angle[2];
            float normalizedDifference;
        };

        BifocalRANSACFilter();
        ~BifocalRANSACFilter();

        void filter(const Match *matches, float *probs, unsigned count, unsigned iterations, LinAlg::Matrix3x3f &bestFundamental, float scaleFactor);
    protected:
        std::vector<float> m_cumulativeProbabilities;
        std::vector<float> m_probabilities;


        void buildProbabilities(float lambda, const Match *matches, unsigned count);
        unsigned pickMatch(float frac) const;


        void doRansac(const Match *matches, unsigned count, unsigned seed, unsigned iters, LinAlg::Matrix3x3f *dstFundamental, float *dstScore);
        void computeMatchProbabilityGivenFourFundamentals(const float *fourFundamentals, const Match *match, __m128 &result);
        void computeProbabilityOfFourFundamentals(const Match *matches, unsigned count, const LinAlg::Matrix3x3f *fundamentals, float *probs);

        float computeMatchProbabilityGivenFundamental(const LinAlg::Matrix3x3f &fundamental, const Match *match);
        void computeProbabilitiesOfFundamental(const Match *matches, unsigned count, const LinAlg::Matrix3x3f &fundamental, float *probs);


};

}

#endif // BIFOCALRANSACFILTER_H
