/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "Camera.h"
#include "InternalCameraCalibration.h"
#include <assert.h>


namespace SFM {


Camera::Camera(InternalCameraCalibration *internalCalibration) : m_internalCalibration(internalCalibration)
{
    m_baCameraHandle = -1;
}

Camera::~Camera()
{
    if (m_baCameraHandle != (unsigned)-1)
        m_ba->removeCamera(m_baCameraHandle);
}

void Camera::setViewMatrix(const LinAlg::Matrix4x4f &viewMatrix)
{
    m_viewMatrix = viewMatrix;
    m_projectionViewMatrix = m_internalCalibration->getProjectionMatrix() * viewMatrix;
    m_projectionViewMatrix3x4 = m_projectionViewMatrix.dropRow(2);

    m_invProjectionViewMatrix = m_projectionViewMatrix;
    m_invProjectionViewMatrix.GaussJordanInvert();

    m_viewMatrix /= m_viewMatrix[3][3];
    m_cameraPosition = (m_viewMatrix.T() * LinAlg::Fill(-m_viewMatrix[0][3], -m_viewMatrix[1][3], -m_viewMatrix[2][3], 0.0f)).StripHom();
}

void Camera::onInternalCameraCalibrationChanged()
{
    setViewMatrix(m_viewMatrix);
}


void Camera::activateBA(BundleAdjustment &ba)
{
    assert(m_baCameraHandle == (unsigned)-1);
    m_baCameraHandle = ba.addCamera(m_internalCalibration->getBAHandle());
    ba.setCamera(m_baCameraHandle, m_viewMatrix);
    m_ba = &ba;
}

void Camera::readBackFromBA(BundleAdjustment &ba)
{
    assert(m_ba == &ba);
    setViewMatrix(ba.getCameraViewMatrix(m_baCameraHandle));
}

void Camera::updateBA(BundleAdjustment &ba)
{
    if (m_baCameraHandle != (unsigned)-1) {
        assert(m_ba == &ba);
        ba.setCamera(m_baCameraHandle, m_viewMatrix);
    }
}


}
