/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "DuplicateTrackRemoval.h"

namespace SFM {

DuplicateTrackRemoval::DuplicateTrackRemoval()
{
    m_hash = std::unique_ptr<SizedHash>(new SizedHash());
}


void DuplicateTrackRemoval::clear()
{
    m_hash->reset();
}

DuplicateTrackRemoval::Element *DuplicateTrackRemoval::mergeOrAdd(const Element &element)
{
    const float minRadius = element.m_size * 0.9f;
    const float maxRadius = element.m_size * 1.2f;

    const float sqrMaxDistance = element.m_size*element.m_size * (2.5f * 2.5f);


    const int minLayer = std::log2(minRadius) + 0.5f;
    const int idealLayer = std::log2(element.m_size) + 0.5f;
    const int maxLayer = std::log2(maxRadius) + 0.5f;

    for (int layer = minLayer; layer <= maxLayer; layer++) {
        float scale = std::exp2(layer + 3.0f); // eight times the radius
        float rcpScale = 1.0f / scale;

        int64_t x = element.m_position[0] * rcpScale;
        int64_t y = element.m_position[1] * rcpScale;
        int64_t z = element.m_position[2] * rcpScale;

        for (int64_t _z = z-1; _z <= z+1; _z++) {
            for (int64_t _y = y-1; _y <= y+1; _y++) {
                for (int64_t _x = x-1; _x <= x+1; _x++) {
                    HashCell &cell = (*m_hash)(_x, _y, _z, layer);

                    for (unsigned i = 0; i < cell.m_elements.reservedSize(); i++) {
                        if (!cell.m_elements.inUse(i)) continue;

                        if (cell.m_elements[i].m_size < minRadius) continue;
                        if (cell.m_elements[i].m_size > maxRadius) continue;

                        if ((cell.m_elements[i].m_position - element.m_position).SQRLen() > sqrMaxDistance) continue;

                        return &cell.m_elements[i];
                    }
                }
            }
        }
    }
    float scale = std::exp2(idealLayer + 3.0f); // eight times the radius
    float rcpScale = 1.0f / scale;

    int64_t x = element.m_position[0] * rcpScale;
    int64_t y = element.m_position[1] * rcpScale;
    int64_t z = element.m_position[2] * rcpScale;
    HashCell &cell = (*m_hash)(x, y, z, idealLayer);

    cell.m_elements.allocate(element);
    return nullptr;
}


}
