/*
    Structure from Motion with Deferred Feature Matching and Subset Bundle Adjustment
    Copyright (C) 2015 Andreas Ley <andy-ley@arcor.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef FRAME_H
#define FRAME_H

#include "Camera.h"
#include <tinyxml.h>
#include "SiftDescriptorDB.h"
#include "../cudaInterface/CudaSift.h"
#include "BundleAdjustment/BundleAdjustment.h"
#include "ObservationGrid.h"
#include <set>
#include "../tools/RasterImage.h"
//#include "../cudaInterface/PatchCache.h"

namespace SFM {

namespace config {
    struct FeatureMatchingConfig;
    struct SubsetSelectionConfig;
}

class InternalCameraCalibration;
class Track;
class DescriptorMatchingStats;

/**
 * @brief Holds the data of a single image for the SfM pipeline.
 * @details This entails, amongst other things, the SIFT descriptor database as well as
 * the camera related data.
 * @ingroup SFMBackend_Group
 */
class Frame
{
    public:
        Frame(InternalCameraCalibration *internalCameraCalibration, const std::string &filename, unsigned frameIndex);
        ~Frame();

        struct ImageFeaturePoint {
            float distortedX, distortedY;
            float undistortedX, undistortedY;
            float lod, angle, size;

            unsigned siftDBIndex;

            float locationPrecision;

#ifdef CudaSift_EXTRACT_PATCH_DATA
            PackedPatch packedPatch;
            PatchCache::SlotHandle patchCacheSlot;
#endif
        };

        inline const std::string &getImageFilename() const { return m_imageFilename; }

        void setInitialFeaturePoints(const CudaSift::FeaturePoint *featurePoints, unsigned count, unsigned width, unsigned height, unsigned numFrames, bool testbenchDontCompile = false);

        void recomputeUndistortedFeaturePointLocations();

        unsigned getNumFeaturePoints() const { return m_featurePoints.size(); }
        const ImageFeaturePoint &getFeaturePoint(unsigned index) const { return m_featurePoints[index]; }
        ImageFeaturePoint &getFeaturePoint(unsigned index) { return m_featurePoints[index]; }

        Camera &getCamera() { return m_camera; }
        const Camera &getCamera() const { return m_camera; }

        inline bool active() const { return m_active; }
        inline void setActive() { m_active = true; }



        void ransacEstimateViewMatrixFromTracks(ChunkedArray<Track> &trackList, LinAlg::Matrix4x4f &viewMatrix, float &score,
                                                const std::vector<SiftDescriptorDB::SiftDescriptor> &globalPatchDescrList,
                                                config::FeatureMatchingConfig &matchingConfig,
                                                DescriptorMatchingStats &descriptorMatchingStats);

        void activate(ChunkedArray<Track> &trackList, const LinAlg::Matrix4x4f &viewMatrix, BundleAdjustment &ba);

        struct NewObservationCandidate {
            unsigned trackIndex;
            Frame *frame;
            unsigned featurePointIndex;
            float minNormDifference;
        };

        void findNewObservations(ChunkedArray<Track> &trackList, std::vector<NewObservationCandidate> &candidates,
                                unsigned timestampClock, DescriptorMatchingStats &descriptorMatchingStats,
                                config::FeatureMatchingConfig &matchingConfig, bool bootstrappingNewImage);

        struct NewTrackCandidate {
            Frame *frame[2];
            unsigned featurePointIndex[2];
        };
        void findNewTracks(ChunkedArray<Track> &trackList, std::vector<std::unique_ptr<Frame> > &frames,
                               BundleAdjustment &ba, std::vector<NewTrackCandidate> &newTrackCandidates,
                               config::FeatureMatchingConfig &matchingConfig,
                               DescriptorMatchingStats &descriptorMatchingStats);

        inline const SiftDescriptorDB &getSiftDescriptorDB() const { return m_siftDescriptorDB; }
        inline ObservationGrid &getObservationGrid() { return m_observationGrid; }


        void dumpGridCoverage(const std::string &filename);

        void voteForBASubset(const config::SubsetSelectionConfig &subsetSelectionConfig);

        inline unsigned getWidth() const { return m_width; }
        inline unsigned getHeight() const { return m_height; }
        inline unsigned getIndex() const { return m_frameIndex; }

        class InterFrameRelation {
            public:
                InterFrameRelation(unsigned numFeaturePoints) {
                    m_featurePointMatchingDry.resize(numFeaturePoints);
                }

                inline void markFeaturePointPairAsTried(unsigned sourceIndex, unsigned destinationIndex) {
                    m_featurePointMatchesAlreadyTried.insert((((uint64_t)sourceIndex) << 32l) | destinationIndex);
                }
                inline bool hasFeaturePointPairBeenTriedBefore(unsigned sourceIndex, unsigned destinationIndex) const {
                    return m_featurePointMatchesAlreadyTried.find((((uint64_t)sourceIndex) << 32l) | destinationIndex) != m_featurePointMatchesAlreadyTried.end();
                }

                inline void markFeaturePointAsDry(unsigned index) {
                    m_featurePointMatchingDry[index] = true;
                }

                inline bool isFeaturePointDry(unsigned index) const {
                    return m_featurePointMatchingDry[index];
                }
            private:
                std::vector<bool> m_featurePointMatchingDry;
                std::set<uint64_t> m_featurePointMatchesAlreadyTried;
        };
    protected:
        unsigned m_width;
        unsigned m_height;
        unsigned m_frameIndex;


        std::vector<InterFrameRelation> m_interFrameRelations;

        std::string m_imageFilename;
        Camera m_camera;
        std::vector<ImageFeaturePoint> m_featurePoints;
        bool m_active;
        SiftDescriptorDB m_siftDescriptorDB;
        ObservationGrid m_observationGrid;
};

}

#endif // FRAME_H
